# Missed Loan Installments Tracking System

## Overview
This implementation adds a comprehensive system to track missed loan installments, suspend users with insufficient funds, and allow admins to manage and resolve missed payments.

## Features Implemented

### 1. Database Structure
- **New Table**: `loan_missed_installments`
  - Tracks each missed installment with details
  - Records due date, missed date, amount, and payment status
  - Links to user and loan records

- **Updated Table**: `user_loans`
  - Added `missed_installment_count` column to track total missed installments

### 2. Automatic Tracking (Cron Job)
**File**: `core/app/Jobs/ProcessLoanInstallments.php`

When a user has insufficient balance:
- Records the missed installment in `loan_missed_installments` table
- Suspends the user account
- Sets suspend reason: "Insufficient funds for Loan Installment"
- Increments `missed_installment_count` on the loan
- Moves to next installment date (doesn't increment `given_installment`)

### 3. Admin Management Interface

#### View Missed Installments
**Route**: `/admin/loan/missed-installments/{loan_id}`
**View**: `resources/views/admin/loan/missed_installments.blade.php`

Displays:
- Loan information
- User details and current balance
- Suspension status
- List of all missed installments (paid and unpaid)
- Total amount due
- Days overdue for each installment

#### Process Payment
**Route**: `POST /admin/loan/pay-missed-installments/{loan_id}`

When admin approves payment:
1. Validates user has sufficient balance
2. Deducts total missed amount from user balance
3. Marks all missed installments as paid
4. Creates transaction records for each payment
5. Creates installment logs
6. Updates loan: increments `given_installment`, adds to `paid_amount`
7. Resets `missed_installment_count` to 0
8. Unsuspends the user
9. Checks if loan should be marked as completed
10. Sends notification to user

#### Manual Unsuspend
**Route**: `POST /admin/loan/unsuspend-user/{loan_id}`

Allows admin to unsuspend user without payment (admin discretion):
- Removes suspension flag
- Clears suspend reason
- Sends notification to user
- Missed installments remain unpaid

### 4. Admin Dashboard Integration

**Updated**: `core/app/Http/Controllers/Admin/LoanController.php`

- Datatables now show missed installment count in red
- Action dropdown includes "Missed Installments" link (shown only when there are missed installments)
- Color-coded display for easy identification

### 5. User Flow Example

**Scenario**: User has 4-month loan, misses months 2 and 3, then pays in month 4

1. **Month 1**: User pays normally ✓
2. **Month 2**: Insufficient balance
   - System records missed installment
   - User suspended
   - Loan continues to month 3
3. **Month 3**: Still insufficient balance
   - Another missed installment recorded
   - User remains suspended
4. **Month 4**: User deposits money
   - Admin views missed installments page
   - Sees 2 unpaid installments
   - Checks user balance is sufficient
   - Clicks "Process Payment & Unsuspend User"
   - System deducts both missed payments
   - User unsuspended
   - Cron job resumes normal processing

## Files Modified/Created

### Created Files:
1. `database/migrations/2025_10_30_000001_create_loan_missed_installments_table.php`
2. `app/Models/LoanMissedInstallment.php`
3. `resources/views/admin/loan/missed_installments.blade.php`

### Modified Files:
1. `app/Jobs/ProcessLoanInstallments.php`
   - Added missed installment tracking
   - Added `recordMissedInstallment()` method

2. `app/Http/Controllers/Admin/LoanController.php`
   - Added `missedInstallments()` method
   - Added `payMissedInstallments()` method
   - Added `unsuspendUser()` method
   - Updated datatables to show missed count

3. `app/Models/UserLoan.php`
   - Added `missed_installment_count` to fillable
   - Added `missedInstallments()` relationship
   - Added `unpaidMissedInstallments()` relationship

4. `routes/admin.php`
   - Added 3 new routes for missed installment management

## Installation Steps

1. **Run Migration**:
   ```bash
   php artisan migrate
   ```

2. **Clear Cache** (if needed):
   ```bash
   php artisan config:clear
   php artisan cache:clear
   php artisan route:clear
   ```

3. **Test the System**:
   - Create a test loan
   - Reduce user balance below installment amount
   - Run cron job or wait for scheduled execution
   - Check admin panel for missed installments

## Admin Usage

1. Navigate to Loans section in admin panel
2. Loans with missed installments will show red badge in "Total Installment" column
3. Click Actions → "Missed Installments" for detailed view
4. Review missed installment details
5. If user has sufficient balance:
   - Click "Process Payment & Unsuspend User"
6. If user doesn't have balance:
   - Wait for user to deposit funds, OR
   - Click "Unsuspend User (Without Payment)" to manually unsuspend

## Benefits

1. **Complete Audit Trail**: Every missed payment is recorded with dates and amounts
2. **Automatic Suspension**: Users are automatically suspended when they miss payments
3. **Flexible Resolution**: Admin can process payments or manually unsuspend
4. **Clear Visibility**: Dashboard shows which loans have issues
5. **Proper Accounting**: All transactions are logged correctly
6. **User Protection**: Cron job continues to run, allowing catch-up payments

## Notes

- Missed installments don't prevent the loan from progressing to the next due date
- User can pay multiple missed installments at once when they have funds
- Admin has full control over suspension/unsuspension
- All actions are logged and create proper transaction records
- System prevents loan completion until all missed installments are paid
