<?php

namespace App\Http\Controllers\Admin;

use Datatables;
use App\Models\User;
use App\Models\UserDps;
use App\Models\Currency;
use App\Models\Transaction;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\InstallmentLog;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\HtmlString;
use App\Http\Controllers\Controller;

class DpsController extends Controller
{
    public function __construct()
    {
    }

    public function datatables(Request $request)
    {
        if ($request->status == 'all') {
            $datas = UserDps::with('lastHistory')->orderBy('id', 'desc')->get();
        } else {
            $datas = UserDps::with('lastHistory')->where('status', $request->status)->orderBy('id', 'desc')->get();
        }

        return Datatables::of($datas)
            ->editColumn('status_last_history', function ($data) {
                if ($data->lastHistory) {
                    $name = e($data->lastHistory->admin->name);
                    $time = $data->lastHistory->created_at;
                    return "{$name}<br><small>{$time}</small>";
                }
            })
            ->editColumn('transaction_no', function (UserDps $data) {
                return '<div>
                    '.$data->transaction_no.'
                    <br>
                    <span class="text-info">'.$data->plan->title.'</span>
                </div>';
            })
            ->editColumn('deposit_amount', function (UserDps $data) {
                $curr = Currency::where('is_default', '=', 1)->first();
                return '<div>
                    '.$curr->sign.$data->deposit_amount.'
                    <br>
                    <span class="text-info">Per Installment '.$curr->sign.$data->per_installment.'</span>
                </div>';
            })
            ->editColumn('user_id', function (UserDps $data) {
                return '<div>
                    <span>'.$data->user->name.'</span>
                    <p>'.$data->user->account_number.'</p>
                </div>';
            })
            ->editColumn('total_installment', function (UserDps $data) {
                $curr = Currency::where('is_default', '=', 1)->first();
                return '<div>
                    '.$data->total_installment.'
                    <br>
                    <span class="text-info">'.$data->given_installment.' Given ('.$curr->sign.$data->paid_amount.')</span>
                </div>';
            })
            ->editColumn('matured_amount', function (UserDps $data) {
                $curr = Currency::where('is_default', '=', 1)->first();
                return '<div>
                    '.$curr->sign.$data->matured_amount.'<br>
                    <span class="text-info">Interest Rate '.$data->interest_rate.' (%)</span>
                </div>';
            })
            ->editColumn('next_installment', function (UserDps $data) {
                return $data->next_installment != null ? Carbon::parse($data->next_installment)->toDateString() : '--';
            })
            ->editColumn('status', function (UserDps $data) {
                $status = match ($data->status) {
                    default => "<span class='badge badge-danger'>Closed</span>",
                    '1', 1 => "<span class='badge badge-info'>Running</span>",
                    '2', 2 => "<span class='badge badge-success'>Matured</span>",
                    '3', 3 => "<span class='badge badge-warning'>Insufficient Balance</span>",
                };

                return new HtmlString($status);
            })
            ->addColumn('action', function (UserDps $data) {
                $curr = Currency::where('is_default', '=', 1)->first();
                $html = '<div class="btn-group mb-1">
                    <button type="button" class="btn btn-primary btn-sm btn-rounded dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                        Actions
                    </button>
                    <div class="dropdown-menu">';

                $html .= '<a href="'.route('admin.dps.log.show', $data->id).'" class="dropdown-item">'.__("Logs").'</a>';
                $html .= '<a href="javascript:void(0);" data-toggle="modal" data-target="#dpsCloseModal" data-href="'.route('admin.dps.close', $data->id).'" class="dropdown-item">'.__("Close").'</a>';

                if ($data->status == 3) {
                    $html .= '<a href="javascript:void(0);" 
                                data-toggle="modal" 
                                data-target="#dpsAddAmountModal" 
                                data-href="'.route('admin.dps.add-amount', $data->id).'" 
                                data-amount="'.$curr->sign.$data->per_installment.'" 
                                class="dropdown-item">'.__("Add Amount").'</a>';
                }

                $html .= '</div></div>';
                return $html;
            })
            ->rawColumns(['status_last_history', 'transaction_no', 'deposit_amount', 'user_id', 'total_installment', 'matured_amount', 'next_installment', 'action'])
            ->toJson();
    }

    public function index()
    {
        $this->installmentCheck();
        return view('admin.dps.index');
    }

    public function running()
    {
        $this->installmentCheck();
        return view('admin.dps.running');
    }

    public function matured()
    {
        return view('admin.dps.matured');
    }


    public function closed()
    {
        return view('admin.dps.closed');
    }

    public function payPendingDps($dpsId)
    {
        try {
            $dps = UserDps::findOrFail($dpsId);

            if ($dps->status != 3) {
                return response()->json('This DPS is not pending payment.');
            }

            $pendingAmount = $dps->per_installment;

            if (($dps->user->balance ?? 0) < $pendingAmount) {
                return response()->json('Insufficient balance. Please deposit money first.');
            }
            
            DB::transaction(function () use ($dps, $pendingAmount) {
                $dps->user->decrement('balance', $pendingAmount);

                $dps->status = 1;
                $dps->paid_amount += $pendingAmount;
                $dps->given_installment += 1;
                $dps->next_installment = Carbon::parse($dps->next_installment)->addDays((int) $dps->installment_interval);
                $dps->save();

                $this->logCreate($dps->transaction_no, $pendingAmount, $dps->user_id);

                Transaction::create([
                    'email' => $dps->user->email,
                    'amount' => $pendingAmount,
                    'type' => 'DPS Installment',
                    'profit' => 'minus',
                    'txnid' => $dps->transaction_no,
                    'user_available_balance' => $dps->user->balance,
                    'user_id' => $dps->user_id,
                ]);
            });
            return response()->json('Pending DPS installment paid successfully.');
        } catch (\Exception $th) {
            return redirect()->back()->with('error', 'Something Went Wrong');
        }
    }


    public function logShow($id)
    {
        $dps = UserDps::with('histories')->findOrfail($id);
        $logs = InstallmentLog::where('transaction_no', $dps->transaction_no)->latest()->paginate(20);
        $currency = Currency::whereIsDefault(1)->first();

        return view('admin.dps.log', compact('dps', 'logs', 'currency'));
    }

    public function installmentCheck()
    {
        $dps = UserDps::whereStatus(1)->get();
        $now = Carbon::now();

        foreach ($dps as $key => $data) {
            if ($data->given_installment == $data->total_installment) {
                if ($data->is_given != 1) {
                    $this->maturedDps($data);
                }
                return false;
            }
            if ($now->gt($data->next_installment)) {
                $this->takeLoanAmount($data->user_id, $data->per_installment);
                $this->logCreate($data->transaction_no, $data->per_installment, $data->user_id);

                $data->next_installment = Carbon::now()->addDays((float)$data->plan->installment_interval);
                $data->given_installment += 1;
                $data->paid_amount += $data->per_installment;
                $data->update();

            }
        }
    }

    public function takeLoanAmount($userId, $installment)
    {
        $user = User::whereId($userId)->first();
        if ($user && $user->balance >= $installment) {
            $user->balance -= $installment;
            $user->update();
        }
    }

    public function maturedDps($data)
    {
        $dps = UserDps::whereId($data->id)->first();
        if ($dps) {
            $dps->status = 2;
            $dps->is_given = 1;
            $dps->next_installment = null;
            $dps->update();

            $trans = new Transaction;
            $trans->amount = $dps->matured_amount;
            $trans->type = 'DPS Matured';
            $trans->profit = 'plus';
            $trans->txnid = $dps->transaction_no;
            $trans->user_id = $dps->user_id;
            $trans->user_available_balance = $dps->user->balance + $dps->matured_amount;
            $trans->save();

            $this->sendMaturedMoney($dps->user_id, $dps->matured_amount);
        }
    }

    public function close(UserDps $dps)
    {
        $msg = 'DPS Closed Successfully.';

        statusHistory($dps->id, 'dps', $dps->status, 0);

        $dps->status = 0;
        $dps->is_given = 1;
        $dps->next_installment = null;
        $dps->save();

        $dpsAmount = round($dps->given_installment * $dps->per_installment, 2);

        $trans = new Transaction;
        $trans->email = $dps->user->email;
        $trans->amount = $dpsAmount;
        $trans->type = 'DPS Closed';
        $trans->profit = 'plus';
        $trans->txnid = $dps->transaction_no;
        $trans->user_id = $dps->user_id;
        $trans->user_available_balance = $dps->user->balance + $dpsAmount;
        $trans->save();

        $this->sendMaturedMoney($dps->user_id, $dpsAmount);
        return response()->json($msg);
    }

    public function sendMaturedMoney($userId, $maturedAmount)
    {
        $user = User::findOrfail($userId);
        if ($user) {
            $user->balance += $maturedAmount;
            $user->update();
        }
    }

    public function logCreate($transactionNo, $amount, $userId)
    {
        $data = new InstallmentLog();
        $data->user_id = $userId;
        $data->transaction_no = $transactionNo;
        $data->type = 'dps';
        $data->amount = $amount;
        $data->save();
    }

}
