<?php

namespace App\Http\Controllers\Admin;

use PDF;
use Datatables;
use Carbon\Carbon;
use App\Models\User;
use App\Models\Currency;
use App\Models\OtherBank;
use App\Models\Beneficiary;
use App\Models\Transaction;
use Illuminate\Http\Request;
use App\Classes\GeniusMailer;
use App\Models\Generalsetting;
use App\Models\BalanceTransfer;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;

class OtherBankTransferController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:admin');
    }

    public function datatables(Request $request)
    {
        $datas = BalanceTransfer::with('user', 'bank', 'lastHistory', 'beneficiary')
        ->whereType('other')
        ->when($request->bank && $request->status !== null && $request->status !== '', function ($query) use ($request) {
            $query->where('other_bank_id', $request->bank)->where('status', $request->status);
        })
        ->when($request->bank && (!$request->status || $request->status === ''), function ($query) use ($request) {
            $query->where('other_bank_id', $request->bank);
        })
        ->when($request->status !== null && $request->status !== '' && !$request->bank, function ($query) use ($request) {
            $query->where('status', $request->status);
        })
        ->orderBy('id', 'desc');

        return Datatables::of($datas)
            ->addColumn('checkbox', function (BalanceTransfer $data) {
                return '<input type="checkbox" class="row-checkbox" value="' . $data->id . '">';
            })
            ->editColumn('user_name', function (BalanceTransfer $data) {
                $data = User::whereId($data->user_id)->first();
                return $data ? $data->name : '';
            })
            ->editColumn('user_ref_name', function (BalanceTransfer $data) {
                $data = Beneficiary::whereId($data->beneficiary_id)->first();
                return $data ? $data->account_name : '';
            })
            ->editColumn('user_id', function (BalanceTransfer $data) {
                $data = User::whereId($data->user_id)->first();
                if ($data) {
                    return '<div>
                                            <span>' . $data->name.' '.$data->last_name . '</span>
                                            <p>' . $data->account_number . '</p>
                                    </div>';
                } else {
                    return $data = '';
                }
            })
            ->editColumn('beneficiary_id', function (BalanceTransfer $data) {
                $data = Beneficiary::whereId($data->beneficiary_id)->first();

                if ($data) {
                    return '<div>
                                            <span>' . $data->account_name . '</span>
                                            <p>' . $data->account_number . '</p>
                                    </div>';
                } else {
                    return $data = '';
                }
            })
            ->editColumn('amount', function (BalanceTransfer $data) {
                $curr = Currency::where('is_default', '=', 1)->first();
                return $curr->sign . $data->amount;
            })
            ->editColumn('bank_name', function ($data) {
                return optional($data->bank)->title;
            })
            ->editColumn('cost', function (BalanceTransfer $data) {
                $curr = Currency::where('is_default', '=', 1)->first();
                return $curr->sign . $data->cost;
            })
            ->editColumn('status_last_history', function ($data) {
                if ($data->lastHistory) {
                    $name = e($data->lastHistory->admin->name);
                    $time = $data->lastHistory->created_at;
                    return "{$name}<br><small>{$time}</small>";
                }
            })
            ->editColumn('status', function (BalanceTransfer $data) {
                if ($data->status == 1) {
                    $status = __('Completed');
                } else if ($data->status == 2) {
                    $status = __('Rejected');
                } else if ($data->status == 3) {
                    $status = __('Processing');
                } else if ($data->status == 4) {
                    $status = __('Hold');
                } else {
                    $status = __('Pending');
                }

                if ($data->status == 1) {
                    $status_sign = 'success';
                } else if ($data->status == 2) {
                    $status_sign = 'danger';
                } else {
                    $status_sign = 'warning';
                }

                return '<div class="btn-group mb-1">
                                <button type="button" class="btn btn-' . $status_sign . ' btn-sm btn-rounded dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                                  ' . $status . '
                                </button>
                                <div class="dropdown-menu" x-placement="bottom-start">
                                  <a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="' . route('admin.other.banks.transfer.status', ['id1' => $data->id, 'status' => 0]) . '">' . __("Pending") . '</a>
                                  <a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="' . route('admin.other.banks.transfer.status', ['id1' => $data->id, 'status' => 3]) . '">' . __("Processing") . '</a>
                                  <a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="' . route('admin.other.banks.transfer.status', ['id1' => $data->id, 'status' => 1]) . '">' . __("Completed") . '</a>
                                  <a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="' . route('admin.other.banks.transfer.status', ['id1' => $data->id, 'status' => 4]) . '">' . __("Hold") . '</a>
                                  <a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="' . route('admin.other.banks.transfer.status', ['id1' => $data->id, 'status' => 2]) . '">' . __("Rejected") . '</a>
                                </div>
                              </div>';
            })
            ->filterColumn('status', function ($query, $keyword) {
                $map = [
                    0 => 'pending',
                    1 => 'completed',
                    2 => 'rejected',
                    3 => 'processing',
                    4 => 'hold',
                ];

                $keyword = strtolower($keyword);

                $query->where(function ($q) use ($map, $keyword) {
                    foreach ($map as $statusNum => $statusText) {
                        if (str_contains($statusText, $keyword)) {
                            $q->orWhere('status', $statusNum);
                        }
                    }
                });
            })
            ->addColumn('action', function (BalanceTransfer $data) {

                return '<div class="btn-group mb-1">
                                  <button type="button" class="btn btn-primary btn-sm btn-rounded dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                                    ' . 'Actions' . '
                                  </button>
                                  <div class="dropdown-menu" x-placement="bottom-start">
                                    <a href="' . route('admin.other.banks.transfer.show', $data->id) . '"  class="dropdown-item">' . __("Details") . '</a>
                                  </div>
                                </div>';
            })
            ->rawColumns(['checkbox', 'user_id', 'beneficiary_id', 'amount', 'cost', 'status', 'action', 'status_last_history'])
            ->toJson();
    }

    public function index()
    {
        $banks = OtherBank::where('status', 1)->get();
        return view('admin.otherbanktransfer.index', compact('banks'));
    }

    public function show($id)
    {
        $data = BalanceTransfer::with('histories')->whereId($id)->first();
        $banefeciary = Beneficiary::whereId($data->beneficiary_id)->first();

        return view('admin.otherbanktransfer.show', compact('data', 'banefeciary'));
    }

    public function status($id1, $id2)
    {
        $data = BalanceTransfer::findOrFail($id1);

        $user = User::findOrFail($data->user_id);
        $authUser =  Auth::guard('admin')->user(); // Logged-in admin or user

        // Check role-based permission
        if (!$authUser->IsSuper() && ($data->status == 1 || $data->status == 2)) {
            $now = Carbon::now();
              $lastUpdate = Carbon::parse($data->updated_at);

              $diff = $lastUpdate->diffInMinutes($now, false);
            if ($diff > 60) {
              //   dd("You do not have permission to change this status.");
                return response()->json( 'You can no longer update this status. 60-minute window has passed.'
                , 200);
            }
        }

        if ($data->status == 0 && $id2 == 0) {
            $msg = 'Already Pending';
            return response()->json($msg);
        }
        if ($data->status == 1 && $id2 == 1) {
            $msg = 'Already Completed';
            return response()->json($msg);
        }
        if ($data->status == 2 && $id2 == 2) {
            $msg = 'Already Rejected';
            return response()->json($msg);
        }
        if ($data->status == 3 && $id2 == 3) {
            $msg = 'Already Processing';
            return response()->json($msg);
        }
        if ($data->status == 4 && $id2 == 4) {
            $msg = 'Already Hold';
            return response()->json($msg);
        }

        $user = User::whereId($data->user_id)->first();

        if ($data->status == 2 && ($id2 == 0 || $id2 == 1|| $id2 == 3 || $id2 == 4)) {
            if ($user) {
                if ($user->balance < $data->final_amount) {
                    $msg = __('User has insufficient balance to revert the rejected transfer.');
                    return response()->json($msg);
                }

                $user->decrement('balance', $data->final_amount);

                $otherBank = OtherBank::whereId($data->other_bank_id)->first();
                $allinfo = [
                    'sender_type' =>  "Revert Rejected",
                    'reciver_type' => $user->account_type ,
                    "fixed_charge_amount" => $otherBank->fixed_charge,
                    "percent_charge" => $otherBank->percent_charge ?? 0,
                    "percent_charge_amount" => ($data->amount / 100) * $otherBank->percent_charge,
                ];

                //trantion added
                $trans = new Transaction();
                $trans->email = $user->email;
                $trans->amount = $data->amount;
                $trans->type = "Send Money Revert Rejected";
                $trans->profit = "minus";
                $trans->txnid = $data->transaction_no;
                $trans->user_id = $user->id;
                // $trans->sender_id = $user->id;
                $trans->charge = $data->cost;
                $trans->user_available_balance = $user->balance;
                $trans->all_info = json_encode($allinfo);
                $trans->save();

                notifyUser($user->id, 'Send Money Revert Rejected',  'tranfer.logs.index', $data->id);
            }
        }elseif($id2 == 1 && $data->status == 2){
            $trans = new Transaction();
            $trans->email = $user->email;
            $trans->amount = $data->amount;
            $trans->type = "Send Money";
            $trans->profit = "minus";
            $trans->txnid = $data->transaction_no;
            $trans->user_id = $user->id;
            $trans->sender_id = $user->id;
            $trans->charge = $data->cost;
            $trans->user_available_balance = $user->balance;
            $trans->save();

            notifyUser($user->id, 'Send Money Confirmed',  'tranfer.logs.index', $data->id);
        }

        if ($id2 == 2) {
            if ($user) {
                $user->increment('balance', $data->final_amount);

                $otherBank = OtherBank::whereId($data->other_bank_id)->first();
                $allinfo = [
                    'sender_type' =>  "Refund(rejeceted)",
                    'reciver_type' => $user->account_type ,
                    "fixed_charge_amount" => $otherBank->fixed_charge,
                    "percent_charge" => $otherBank->percent_charge ?? 0,
                    "percent_charge_amount" => ($data->amount / 100) * $otherBank->percent_charge,
                ];

                //trantion added
                $trans = new Transaction();
                $trans->email = $user->email;
                $trans->amount = $data->amount;
                $trans->type = "Send Money Refund";
                $trans->profit = "plus";
                $trans->txnid = $data->transaction_no;
                $trans->user_id = $user->id;
                $trans->sender_id = $user->id;
                $trans->charge = $data->cost;
                $trans->user_available_balance = $user->balance;
                $trans->all_info = json_encode($allinfo);
                $trans->save();

                notifyUser($user->id, 'Send Money Rejected',  'tranfer.logs.index', $data->id);
            }
        }
        
        statusHistory($data->id, 'other_bank', $data->status, $id2);

        if ($id2 == 1 && $data->status != 1) {
            $trans = Transaction::where('txnid', $data->transaction_no)->first();
            // $trans->charge = $data->cost;
            $trans->save();
            notifyUser($user->id, 'Send Money Confirmed',  'tranfer.logs.index', $data->id);
        }

        $data->status = $id2;
        $data->update();

        if($data->status == 1){
            $gs = Generalsetting::first();
            if ($gs->is_smtp == 1) {
                $emailData = [
                    'to' => $data->user->email,
                    'type' => "other bank",
                    'cname' => $data->user->name . '' . $data->user->last_name,
                    'oamount' => $data->amount,
                    'aname' => "",
                    'aemail' => "",
                    'wtitle' => "",
                ];

                $mailer = new GeniusMailer();
                $mailer->sendAutoMail($emailData);
            } else {
                $to = $data->user->email;
                $subject = " Wire Transfer Confirmed.";
                $msg = "Hello " . $data->user->name . '' . $data->user->last_name . "!\nWire Transfer Confirmed.\nThank you.";
                $headers = "From: " . $gs->from_name . "<" . $gs->from_email . ">";
                mail($to, $subject, $msg, $headers);
            }
        }

        $msg = __('Status Updated Successfully.');
        return response()->json($msg);
    }

    public function pdfPrint(Request $request)
    {
        $gs = Generalsetting::first();
        $ids = $request->input('ids');
        $curr = Currency::where('is_default', '=', 1)->first();
        $data = [
            'title' => $gs->title,
            'date' => date('Y-m-d'),
            'currency' => $curr,
        ];

        if ($ids) {
            $data['transfers'] = BalanceTransfer::with('user', 'bank', 'lastHistory', 'beneficiary')
                ->whereType('other')
                ->whereIn('id', $ids)
                ->orderBy('id', 'desc')
                ->get();
        } else {
            $data['transfers'] = BalanceTransfer::with('user', 'bank', 'lastHistory', 'beneficiary')
                ->whereType('other')
                ->orderBy('id', 'desc')
                ->get();
        }
        $pdf = PDF::loadView('admin.user.otherBankTransferpdf', $data);

        return $pdf->stream(uniqid('withdraw-') . '.pdf');
    }


    public function bulkChangeOtherBankTransferStatus(Request $request)
    {
        $ids = $request->input('ids');
        $newStatus = $request->input('status');
        $updatedCount = 0;
        $skippedCount = 0;
        $messages = [];

        if (!is_array($ids) || empty($ids)) {
            return response()->json('No transfer IDs provided.', 400);
        }

        $authUser = Auth::guard('admin')->user();
        $gs = Generalsetting::first();

        foreach ($ids as $id) {
            $data = BalanceTransfer::find($id);
            if (!$data) continue; // Skip invalid IDs

            if ($data->status == 0 && $newStatus == 0) {
                $messages[] = "Transaction {$data->transaction_no} already Pending.";
                continue;
            }
            if ($data->status == 1 && $newStatus == 1) {
                $messages[] = "Transaction {$data->transaction_no} already Completed.";
                continue;
            }
            if ($data->status == 2 && $newStatus == 2) {
                $messages[] = "Transaction {$data->transaction_no} already Rejected.";
                continue;
            }
            if ($data->status == 3 && $newStatus == 3) {
                $messages[] = "Transaction {$data->transaction_no} already Processing.";
                continue;
            }
            if ($data->status == 4 && $newStatus == 4) {
                $messages[] = "Transaction {$data->transaction_no} already Hold.";
                continue;
            }



            $user = User::find($data->user_id);
            if (!$user) continue;

            if (!$authUser->IsSuper() && ($data->status == 1 || $data->status == 2)) {
                $lastUpdate = Carbon::parse($data->updated_at);
                $diff = $lastUpdate->diffInMinutes(now(), false);
                if ($diff > 60) {
                    $skippedCount++;
                    $messages[] = "Transaction {$data->transaction_no} skipped — 60-minute window passed.";
                    continue;
                }
            }

            if (in_array($newStatus, [2])) {
                $user->increment('balance', $data->final_amount);

                $otherBank = OtherBank::find($data->other_bank_id);
                if ($otherBank) {
                    $allinfo = [
                        'sender_type' => "Refund(rejected)",
                        'reciver_type' => $user->account_type,
                        'fixed_charge_amount' => $otherBank->fixed_charge,
                        'percent_charge' => $otherBank->percent_charge ?? 0,
                        'percent_charge_amount' => ($data->amount / 100) * $otherBank->percent_charge,
                    ];

                    $trans = new Transaction();
                    $trans->email = $user->email;
                    $trans->amount = $data->amount;
                    $trans->type = "Send Money Refund";
                    $trans->profit = "plus";
                    $trans->txnid = $data->transaction_no;
                    $trans->user_id = $user->id;
                    $trans->charge = $data->cost;
                    $trans->user_available_balance = $user->balance;
                    $trans->all_info = json_encode($allinfo);
                    $trans->save();
                }
            }

            statusHistory($data->id, 'other_bank', $data->status, $newStatus);

        
            if ($newStatus == 1 && $data->status == 2) {
                $trans = new Transaction();
                $trans->email = $user->email;
                $trans->amount = $data->amount;
                $trans->type = "Send Money";
                $trans->profit = "minus";
                $trans->txnid = $data->transaction_no;
                $trans->user_id = $user->id;
                // $trans->sender_id = $user->id;
                $trans->charge = $data->cost;
                $trans->user_available_balance = $user->balance;
                $trans->save();
            }
            $data->status = $newStatus;
            $data->update();

            if ($data->status  == 1) {
                if ($gs->is_smtp == 1) {
                    $emailData = [
                        'to' => $user->email,
                        'type' => "other bank",
                        'cname' => $user->name . ' ' . $user->last_name,
                        'oamount' => $data->amount,
                        'aname' => "",
                        'aemail' => "",
                        'wtitle' => "",
                    ];

                    $mailer = new GeniusMailer();
                    $mailer->sendAutoMail($emailData);
                } else {
                    $to = $user->email;
                    $subject = "Wire Transfer Confirmed.";
                    $msg = "Hello " . $user->name . ' ' . $user->last_name . "!\nWire Transfer Confirmed.\nThank you.";
                    $headers = "From: " . $gs->from_name . "<" . $gs->from_email . ">";
                    mail($to, $subject, $msg, $headers);
                }
            }

            $updatedCount++;
        }

        $summaryMsg = "{$updatedCount} record(s) updated successfully.";
        if ($skippedCount > 0) {
            $summaryMsg .= " {$skippedCount} record(s) skipped due to 60-minute limit.";
        }

        return response()->json([
            'message' => $summaryMsg,
            'notes' => $messages,
        ]);
    }
}
