<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Currency;
use App\Models\PaymentGateway;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Throwable;
use Auth;
use Yajra\DataTables\Facades\DataTables;

class PaymentGatewayController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth:admin']);
    }

    private function setEnv($key, $value, $prev)
    {
        file_put_contents(app()->environmentFilePath(), str_replace(
            $key.'='.$prev,
            $key.'='.$value,
            file_get_contents(app()->environmentFilePath())
        ));
    }

    public function datatables()
    {
        $datas = PaymentGateway::orderBy('sorting_order', 'asc');
        return DataTables::of($datas)
            ->addIndexColumn()
            ->filterColumn('title', function ($query, $keyword) {
                if ($keyword) {
                    $query->where('title', 'LIKE', "%{$keyword}%")
                        ->orWhere('name', 'LIKE', "%{$keyword}%");
                }
            })
            ->filterColumn('currency', function ($query, $keyword) {
                if ($keyword) {
                    $currencies = Currency::where('name', 'LIKE', "%{$keyword}%")->pluck('id')->toArray();
                    foreach ($currencies as $currency_id) {
                        $query->orWhereRaw("JSON_CONTAINS(currency_id, '\"{$currency_id}\"')");
                        $query->orWhereRaw("JSON_CONTAINS(currency_id, '{$currency_id}')");
                    }
                }
            })
            ->editColumn('title', function (PaymentGateway $data) {
                if ($data->type == 'automatic') {
                    return $data->name;
                } else {
                    return $data->title;
                }
            })
            ->addColumn('logo', function (PaymentGateway $data) {
                if (!empty($data->logo)) {
                    return '<img src="'.url('assets/images/'.$data->logo).'" alt="Image" width="50">';
                }
                return '';
            })
            ->addColumn('currency', function (PaymentGateway $data) {
                $currency_id = json_decode($data->currency_id, true);
                $currency_id = array_map('intval', $currency_id);
                $currencies = Currency::whereIn('id', $currency_id)->get();

                try {
                    $currencies->pluck('name')->join(', ');
                } catch (Throwable $e) {
                    throw $e;
                }

                return $currencies->pluck('name')->join(', ');
            })
            ->editColumn('fund_available', function (PaymentGateway $data) {
                return number_format(round($data->fund_available ?? 0), 2);
            })
            ->editColumn('status', function (PaymentGateway $data) {
                $status = $data->status == 1 ? __('Activated') : __('Deactivated');
                $status_sign = $data->status == 1 ? 'success' : 'danger';

                return '<div class="btn-group mb-1">
                    <button type="button" class="btn btn-'.$status_sign.' btn-sm btn-rounded dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                      '.$status.'
                    </button>
                    <div class="dropdown-menu" x-placement="bottom-start">
                      <a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="'.route('admin.payment.status', ['id1' => $data->id, 'id2' => 1]).'">'.__("Activate").'</a>
                      <a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="'.route('admin.payment.status', ['id1' => $data->id, 'id2' => 0]).'">'.__("Deactivate").'</a>
                    </div>
                  </div>';
            })
            ->editColumn('withdraw_status', function (PaymentGateway $data) {
                $status = $data->withdraw_status == 1 ? __('Activated') : __('Deactivated');
                $status_sign = $data->withdraw_status == 1 ? 'success' : 'danger';

                return '<div class="btn-group mb-1">
                                <button type="button" class="btn btn-'.$status_sign.' btn-sm btn-rounded dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                                  '.$status.'
                                </button>
                                <div class="dropdown-menu" x-placement="bottom-start">
                                  <a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="'.route('admin.payment.wstatus', ['id1' => $data->id, 'id2' => 1]).'">'.__("Activate").'</a>
                                  <a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="'.route('admin.payment.wstatus', ['id1' => $data->id, 'id2' => 0]).'">'.__("Deactivate").'</a>
                                </div>
                              </div>';
            })
                   ->editColumn('bank_plan_status', function (PaymentGateway $data) {
                $status = $data->bank_plan_status == 1 ? __('Activated') : __('Deactivated');
                $status_sign = $data->bank_plan_status == 1 ? 'success' : 'danger';

                return '<div class="btn-group mb-1">
                                <button type="button" class="btn btn-'.$status_sign.' btn-sm btn-rounded dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                                  '.$status.'
                                </button>
                                <div class="dropdown-menu" x-placement="bottom-start">
                                  <a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="'.route('admin.payment.bankplanstatus', ['id1' => $data->id, 'id2' => 1]).'">'.__("Activate").'</a>
                                  <a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="'.route('admin.payment.bankplanstatus', ['id1' => $data->id, 'id2' => 0]).'">'.__("Deactivate").'</a>
                                </div>
                              </div>';
            })
            ->addColumn('action', function (PaymentGateway $data) {
                // $editLink = route('admin.payment.edit', $data->id);
                // $deleteLink = route('admin.payment.delete', $data->id);
                // $deleteButton = '<button type="button" data-toggle="modal" data-target="#deleteModal"  data-href="'.$deleteLink.'" class="btn btn-danger btn-sm btn-rounded ml-2">
                //     <i class="fas fa-trash"></i>
                // </button>';
                // $delete = $data->type == 'automatic' || $data->keyword != null ? "" : $deleteButton;
                // return '<div class="d-inline-flex"><a href="'.$editLink.'" class="btn btn-primary btn-sm btn-rounded">
                //         <i class="fas fa-edit"></i></a>'.$delete.'</div>';

                    if (Auth::guard('admin')->user()->IsSuper()) {
        $editLink = route('admin.payment.edit', $data->id);
        $deleteLink = route('admin.payment.delete', $data->id);
        $deleteButton = '<button type="button" data-toggle="modal" data-target="#deleteModal" data-href="'.$deleteLink.'" class="btn btn-danger btn-sm btn-rounded ml-2">
            <i class="fas fa-trash"></i>
        </button>';
        $delete = $data->type == 'automatic' || $data->keyword != null ? "" : $deleteButton;
        return '<div class="d-inline-flex"><a href="'.$editLink.'" class="btn btn-primary btn-sm btn-rounded">
                <i class="fas fa-edit"></i></a>'.$delete.'</div>';
    } else {
        // Not authenticated as admin, show nothing in action column
        return '';
    }
            })
            ->rawColumns(['status', 'withdraw_status', 'bank_plan_status', 'action', 'logo'])
            ->toJson();
    }


    public function index()
    {
        $data['currencies'] = Currency::orderBy('id', 'desc')->get();
        return view('admin.payment.index', $data);
    }

    public function create()
    {
        $data['currencies'] = Currency::orderBy('id', 'desc')->get();
        return view('admin.payment.create', $data);
    }

    public function store(Request $request)
    {
        $rules = [
            'title' => 'unique:payment_gateways',
            'logo' => 'nullable|file|mimes:jpeg,jpg,png,gif|max:1024',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return response()->json(array('errors' => $validator->getMessageBag()->toArray()));
        }

        $validated = $this->validateAdditionalFields($request);
        if (isset($validated['fails']) && $validated['fails']) {
            return response()->json(array('errors' => $validated['errors']));
        }

        $data = new PaymentGateway();
        $input = $request->except(['_token', 'form_builder']);

        if ($request->hasFile('logo')) {
            $file = $request->file('logo');
            $name = strtolower("pg-".Str::slug($request->title."-".Str::random(8)).".".$file->extension());
            $file->move('assets/images/', $name);
            $input['logo'] = $name;
        }



        $input['deposit_charge'] = $request->deposit_charge ?? 0;
        $input['deposit_charge_fix'] = $request->deposit_charge_fix ?? 0;
        $input['type'] = "manual";
        $input['additional_fields'] = $validated['fields'] ?? [];
        $input['currency_id'] = json_encode([$request->currency_id]);
        $input['withdraw_status'] = empty($validated['withdraw_status']);
        $input['sorting_order'] = PaymentGateway::max('sorting_order') + 1;
        $data->fill($input)->save();

        $msg = __('New Data Added Successfully.').' '.'<a href="'.route("admin.payment.index").'">'.__('View Lists.').'</a>';
        return response()->json($msg);
    }

    public function edit($id)
    {
        $data['currencies'] = Currency::orderBy('id', 'desc')->get();
        $pg = PaymentGateway::findOrFail($id);
        $data['additionalFields'] = $pg->additional_fields ?? [];
        $currency_id = json_decode($pg->currency_id, true);
        $data['currency_id'] = array_map('intval', $currency_id);
        $data['data'] = $pg;
        return view('admin.payment.edit', $data);
    }

    public function update(Request $request, $id)
    {

        $data = PaymentGateway::findOrFail($id);
        $prev = '';

        if ($data->type == "automatic") {
            if (PaymentGateway::where('name', $request->name)->where('id', '!=', $id)->exists()) {
                return response()->json(array('errors' => [0 => 'This name has already been taken.']));
            }

            $validated = $this->validateAdditionalFields($request);
            if (isset($validated['fails']) && $validated['fails']) {
                return response()->json(array('errors' => $validated['errors']));
            }

            $input = $request->except(['_token', 'form_builder']);

            $info_data = $input['pkey'];

            if ($data->keyword == 'mollie') {
                $paydata = $data->convertAutoData();
                $prev = $paydata['key'];
            }
            if ($file = $request->file('photo')) {
                $paydata = $data->convertAutoData();
                $name = Str::random(8).time().'.'.$file->getClientOriginalExtension();
                $data->upload($name, $file, $paydata['photo']);
                $info_data['photo'] = $name;
            } else {
                if (strpos($data->information, 'photo') !== false) {
                    $paydata = $data->convertAutoData();
                    $info_data['photo'] = $paydata['photo'];
                }
            }

            if (array_key_exists("sandbox_check", $info_data)) {
                $info_data['sandbox_check'] = 1;
            } else {
                if (strpos($data->information, 'sandbox_check') !== false) {
                    $info_data['sandbox_check'] = 0;
                    $text = $info_data['text'];
                    unset($info_data['text']);
                    $info_data['text'] = $text;
                }
            }

            if ($request->hasFile('logo')) {
                $file = $request->file('logo');
                $name = strtolower("pg-".Str::slug($request->name."-".Str::random(8)).".".$file->extension());
                $file->move('assets/images/', $name);
                $input['logo'] = $name;
                if (Storage::exists('assets/images/'.$data->logo)) {
                    Storage::delete('assets/images/'.$data->logo);
                }
            }

            $input['withdraw_status'] = empty($validated['withdraw_status']);
            $input['additional_fields'] = $validated['fields'] ?? [];
              $input['deposit_charge'] = $request->deposit_charge ?? 0;
                $input['deposit_charge_fix'] = $request->deposit_charge_fix ?? 0;
            $input['information'] = json_encode($info_data);
            $data->update($input);

            if ($data->keyword == 'mollie') {
                $paydata = $data->convertAutoData();
                $this->setEnv('MOLLIE_KEY', $paydata['key'], $prev);
            }
        } else {
            $rules = [
                'title' => 'unique:payment_gateways,title,'.$id,
                'logo' => 'nullable|file|mimes:jpeg,jpg,png,gif|max:1024',
            ];

            $validator = Validator::make($request->all(), $rules);
            if ($validator->fails()) {
                return response()->json(array('errors' => $validator->getMessageBag()->toArray()));
            }

            $validated = $this->validateAdditionalFields($request);
            if (isset($validated['fails']) && $validated['fails']) {
                return response()->json(array('errors' => $validated['errors']));
            }

            $input = $request->except(['_token', 'form_builder']);
            if ($request->hasFile('logo')) {
                $file = $request->file('logo');
                $name = strtolower("pg-".Str::slug($request->title."-".Str::random(8)).".".$file->extension());
                $file->move('assets/images/', $name);
                $input['logo'] = $name;
                if (Storage::exists('assets/images/'.$data->logo)) {
                    Storage::delete('assets/images/'.$data->logo);
                }
            }

            $input['currency_id'] = json_encode([$request->currency_id]);
            $input['withdraw_status'] = empty($validated['withdraw_status']);
            $input['additional_fields'] = $validated['fields'] ?? [];
              $input['deposit_charge'] = $request->deposit_charge ?? 0;
            $data->update($input);
        }

        $msg = __('Data Updated Successfully.').' '.'<a href="'.route("admin.payment.index").'">'.__('View Lists.').'</a>';
        return response()->json($msg);
    }

    function validateAdditionalFields($request)
    {
        $data = [];
        $formBuilder = $request->get('form_builder', []);
        if (!empty($formBuilder)) {
            $fieldsRules = [];
            $fieldsMessages = [];
            foreach ($formBuilder as $key => $field) {
                $fieldsRules["$key.field_name"] = 'required';
                $fieldsMessages["$key.field_name"] = 'The additional field name is required.';
                $data['fields'][] = $field;
            }
            $validator = Validator::make($formBuilder, $fieldsRules, $fieldsMessages);
            if ($validator->fails()) {
                $data['fails'] = true;
                $data['errors'] = $validator->getMessageBag()->toArray();
            }
        }

        return $data;
    }


    public function status($id1, $id2)
    {
        $data = PaymentGateway::findOrFail($id1);
        $data->status = $id2;
        $data->update();

        $msg = __('Status Updated Successfully.');
        return response()->json($msg);
    }


    public function wstatus($id1, $id2)
    {
        $data = PaymentGateway::findOrFail($id1);
        $data->withdraw_status = $id2;
        $data->update();

        $msg = __('Withdraw Status Updated Successfully.');
        return response()->json($msg);
    }

        public function bankplanstatus($id1, $id2)
    {
        $data = PaymentGateway::findOrFail($id1);
        $data->bank_plan_status = $id2;
        $data->update();

        $msg = __('Withdraw Status Updated Successfully.');
        return response()->json($msg);
    }

    public function destroy($id)
    {
        $data = PaymentGateway::findOrFail($id);
        if ($data->type == 'manual' || $data->keyword != null) {
            $data->delete();
        }

        $msg = __('Data Deleted Successfully.');
        return response()->json($msg);
    }

    public function sort()
    {
        return view('admin.payment.sort', [
            'items' => PaymentGateway::orderBy('sorting_order', 'asc')->get()
        ]);
    }

    public function sortUpdate(Request $request)
    {
        $ids = $request->ids;
        $count = 1;

        foreach ($ids as $id) {
            $data = PaymentGateway::findOrFail($id);
            $data->sorting_order = $count++;
            $data->save();
        }

        return response()->json(['success' => true, 'message' => __('Payment gateways sorted successfully.')]);
    }
}
