<?php

namespace App\Http\Controllers;

use App\Models\Currency;
use App\Models\PaymentMerchant;
use App\Models\PaymentRequest;
use App\Models\PaymentTransactions;
use App\Models\Transaction;
use App\Models\User;
use Endroid\QrCode\Color\Color;
use Endroid\QrCode\Encoding\Encoding;
use Endroid\QrCode\ErrorCorrectionLevel;
use Endroid\QrCode\QrCode;
use Endroid\QrCode\RoundBlockSizeMode;
use Endroid\QrCode\Writer\SvgWriter;
use File;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Throwable;

class PaymentController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth')->only(['process']);
    }

    public function payment()
    {
        return view('payment.payment');
    }

    public function paymentRequest(Request $request)
    {
        $hasError = false;
        $errorMessage = "";

        if (empty(URL::previous())) {
            $hasError = true;
            $errorMessage = "Invalid Request";
        }

        if (!$hasError && empty($request->token)) {
            $hasError = true;
            $errorMessage = "Invalid Request: Token was not provided";
        }

        $requiredInput = [
            "merchant_id",
            "amount",
            "currency",
            "customer.name",
            "customer.email",
            "return_url",
            "cancel_url",
        ];

        $missingInput = [];
        foreach ($requiredInput as $key) {
            if (!Arr::get($request->all(), $key)) {
                $missingInput[] = $key;
            }
        }
        if (!$hasError && !empty($missingInput)) {
            $hasError = true;
            $errorMessage = "Missing required parameter: ".implode(", ", $missingInput);
        }

        $merchant = PaymentMerchant::where(['merchant_id' => $request->merchant_id])->first();

        if (!$hasError && !$merchant) {
            $hasError = true;
            $errorMessage = "Invalid Merchant Information";
        }

        if (!$hasError && $merchant->public_key != $request->token) {
            $hasError = true;
            $errorMessage = "Invalid Merchant Information";
        }

        if (!$hasError && $merchant->status == 0) {
            $hasError = true;
            $errorMessage = "Merchant is not receiving payments";
        }

        $currencies = Currency::where('status', 1)->pluck('name')->map('strtoupper')->all();

        if (!$hasError && !in_array(strtoupper($request->currency), $currencies)) {
            $hasError = true;
            $errorMessage = "Unsupported currency: ".strtoupper($request->currency);;
        }

        if ($hasError) {
            return view('payment.invalid', [
                'message' => $errorMessage,
            ]);
        }

        $paymentRequest = new PaymentRequest();
        $paymentRequest->fill($request->except('token'));
        $paymentRequest->customer_data = $request->customer ?? [];
        $paymentRequest->referrer = URL::previous();
        $paymentRequest->request_id = "req_".Str::random(20);
        $paymentRequest->save();

        $redirectUrl = route('payment.payment', [$request->merchant_id, $paymentRequest->id]);

        if ($request->ajax() || $request->wantsJson()) {
            if ($hasError) {
                return [
                    'message' => $errorMessage,
                    'status' => 'error'
                ];
            }

            return [
                'redirect_to' => $redirectUrl,
            ];
        }

        return redirect($redirectUrl);
    }

    public function paymentQrCode($paymentRequestId)
    {
        $qrCodeWriter = new SvgWriter();
        $qrCode = new QrCode(
            data: encrypt($paymentRequestId),
            encoding: new Encoding('UTF-8'),
            errorCorrectionLevel: ErrorCorrectionLevel::High,
            size: 200,
            margin: 0,
            roundBlockSizeMode: RoundBlockSizeMode::Margin,
            foregroundColor: new Color(0, 0, 0),

        );

        $qrCodeResult = $qrCodeWriter->write($qrCode);

        File::ensureDirectoryExists(public_path('assets/images/payment'));
        $qrCodePath = "assets/images/payment/qr-code-$paymentRequestId.svg";
        $qrCodeResult->saveToFile($qrCodePath);

        return url($qrCodePath);
    }

    public function index($merchantId, $paymentRequestId)
    {
        $merchant = PaymentMerchant::where(['merchant_id' => $merchantId])->firstOr(fn() => abort(403));

        $paymentRequest = PaymentRequest::where(['merchant_id' => $merchantId, 'id' => $paymentRequestId])->firstOr(fn() => abort(403));

        $qrCodeUrl = $this->paymentQrCode($paymentRequestId);

        return view('payment.index', [
            'paymentRequest' => $paymentRequest,
            'merchant' => $merchant,
            'qrCodeUrl' => $qrCodeUrl,
            'user' => auth()->user(),
        ]);
    }

    public function login(Request $request, $paymentRequestId)
    {
        $paymentRequest = PaymentRequest::findOr($paymentRequestId, fn() => abort(403));

        try {
            $request->validate([
                'email' => 'required|email',
                'password' => 'required|string',
            ]);

            $credentials = $request->only('email', 'password');
            if (auth()->attempt($credentials)) {
                $request->session()->regenerate();
                return view('payment.fragments.process-payment', [
                    'user' => auth()->user(),
                    'paymentRequest' => $paymentRequest,
                ]);
            }
            return back()->withErrors([
                'email' => 'The provided credentials do not match our records.',
            ]);
        } catch (ValidationException $th) {
            return view('payment.fragments.login', [
                'paymentRequest' => $paymentRequest
            ])->withErrors($th->errors());
        }
    }

    /**
     * @throws Throwable
     */
    public function process(Request $request, $paymentRequestId)
    {
        $paymentRequest = PaymentRequest::findOr($paymentRequestId, fn() => abort(403));
        $merchant = PaymentMerchant::where(['merchant_id' => $paymentRequest->merchant_id])->firstOr(fn() => abort(403));
        $merchantUser = User::find($merchant->user_id);

        $plan = auth()->user()->bankPlan;

        $feeRate = 2;
        if ($plan) {
            $feeRate = $plan->payment_processing_fee_rate;
        }

        $feeAmount = round($paymentRequest->amount * $feeRate / 100, 2);

        $paymentData = $paymentRequest->only([
            'merchant_id',
            'amount',
            'currency',
            'customer_data',
            'metadata',
            'description',
        ]);

        try {
            DB::beginTransaction();

            $user = auth()->user();

            if ($user->balance < $paymentRequest->amount) {
                $urlQuery = http_build_query([
                    'status' => 'failed',
                    'message' => 'Insufficient balance',
                ]);
                $url = rtrim($paymentRequest->webhook_url, '/')."?$urlQuery";

                return redirect(url($url));
            }

            $transaction = PaymentTransactions::create([
                ...$paymentData,
                'payment_link_id' => $paymentRequest->id,
                'payer_id' => $user->id,
                'processed_at' => now(),
                'fee_amount' => $feeAmount,
                'reference' => '',
                'status' => 'pending'
            ]);


            if ($transaction) {
                $transaction->update(['status' => 'completed']);

                $user->decrement('balance', $paymentRequest->amount);

                $trans = new Transaction();
                $trans->email = $merchantUser->email;
                $trans->amount = $paymentRequest->amount;
                $trans->type = "Payment";
                $trans->profit = "plus";
                $trans->txnid = Str::random(12);
                $trans->user_id = $merchant->user_id;
                $trans->charge = $feeAmount;
                $trans->save();

                $transaction->update(['reference' => $trans->txnid]);

                $trans = new Transaction();
                $trans->email = $user->email;
                $trans->amount = $paymentRequest->amount;
                $trans->type = "Payment";
                $trans->profit = "minus";
                $trans->txnid = Str::random(12);
                $trans->user_id = $user->id;
                $trans->save();

                $response = [
                    'status' => 'complete',
                    'payment_id' => $transaction->id,
                    'request_id' => $paymentRequest->id,
                ];

                try {
                    Http::post($paymentRequest->webhook_url, $response);
                } catch (Throwable $th) {
                    Log::error($th);
                }
            }

            DB::commit();

            $urlQuery = http_build_query($response);
            $url = rtrim($paymentRequest->return_url, '/')."?$urlQuery";

            return redirect(url($url));
        } catch (Throwable $th) {
            DB::rollBack();
            Log::error($th);

            $urlQuery = http_build_query(['status' => 'failed']);
            $url = rtrim($paymentRequest->return_url, '/')."?$urlQuery";

            return redirect(url($url));
        }

    }

    public function cancel(string $requestId)
    {
        $paymentRequest = PaymentRequest::findOr($requestId, fn() => abort(403));

        return redirect($paymentRequest->cancel_url);
    }
}
