<?php

namespace App\Jobs;

use Carbon\Carbon;
use App\Models\UserDps;
use App\Models\Transaction;
use App\Models\InstallmentLog;
use Illuminate\Support\Facades\DB;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;

class DpsCollectionJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function handle(): void
    {
        logger('Dps Working');
        $now = Carbon::now();

        // Get all active DPS
        $dpses = UserDps::where('status', 1)->get();

        foreach ($dpses as $dps) {
            DB::transaction(function () use ($dps, $now) {
                $user = $dps->user;

                if ($dps->next_installment && $dps->next_installment <= $now && $dps->given_installment < $dps->total_installment) {

                    if ($user->balance >= $dps->per_installment) {
                        $user->decrement('balance', $dps->per_installment);

                        // Log installment
                        $log = new InstallmentLog();
                        $log->user_id = $user->id;
                        $log->transaction_no = $dps->transaction_no;
                        $log->type = 'dps';
                        $log->amount = $dps->per_installment;
                        $log->save();

                        // Transaction
                        $trans = new Transaction();
                        $trans->email = $user->email;
                        $trans->amount = $dps->per_installment;
                        $trans->type = "DPS Installment";
                        $trans->profit = "minus";
                        $trans->txnid = $dps->transaction_no;
                        $trans->user_available_balance = $user->balance;
                        $trans->user_id = $user->id;
                        $trans->save();

                        $dps->paid_amount += $dps->per_installment;
                        $dps->given_installment += 1;
                        $dps->next_installment = Carbon::parse($dps->next_installment)->addDays((float)$dps->installment_interval);
                        $dps->save();

                    } else {
                        $dps->status = 3; // 3 = insufficient balance
                        $dps->save();
                    }
                }

                // -----------------------------
                // 2. Mature DPS after last installment + 1 month
                // -----------------------------
                $lastInstallmentDate = Carbon::parse($dps->next_installment)->subDays($dps->installment_interval);
                $matureDate = $lastInstallmentDate->addMonth();

                if ($dps->given_installment >= $dps->total_installment && $matureDate <= $now && $dps->status != 2) {
                    $principal = $dps->deposit_amount;
                    $interest = $dps->matured_amount - $dps->deposit_amount;

                    if ($interest > 0) {
                        $transProfit = new Transaction();
                        $transProfit->email = $user->email;
                        $transProfit->amount = $interest;
                        $transProfit->type = "DPS Interest";
                        $transProfit->profit = "plus";
                        $transProfit->txnid = $dps->transaction_no;
                        $transProfit->user_id = $user->id;
                        $transProfit->user_available_balance = $user->balance + $interest;
                        $transProfit->save();

                        $user->balance += $interest;
                    }

                    // Pay principal
                    $transPrincipal = new Transaction();
                    $transPrincipal->email = $user->email;
                    $transPrincipal->amount = $principal;
                    $transPrincipal->type = "DPS Principal Return";
                    $transPrincipal->profit = "plus";
                    $transPrincipal->txnid = $dps->transaction_no;
                    $transPrincipal->user_id = $user->id;
                    $transPrincipal->user_available_balance = $user->balance + $principal;
                    $transPrincipal->save();

                    $user->balance += $principal;
                    $user->save();

                    $dps->status = 2;
                    $dps->save();
                }
            });
        }
    }
}