<?php

namespace App\Jobs;

use Carbon\Carbon;
use App\Models\UserFdr;
use App\Models\Transaction;
use Illuminate\Support\Facades\DB;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;

class MaturedFdrProfit implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;


    public function handle(): void
    {
        logger('FDR Working');
        $now = Carbon::now();

        // Get all active FDRs
        $fdrs = UserFdr::where('status', 1)->get();

        foreach ($fdrs as $fdr) {
            DB::transaction(function () use ($fdr, $now) {
                $user = $fdr->user;
                $planType = $fdr->profit_type;

                if ($planType === 'partial') {
                    if ($fdr->next_profit_time && $fdr->next_profit_time <= $now) {

                        $totalDays = (float) $fdr->plan->matured_days;
                        $intervalDays = (float) $fdr->plan->interest_interval;
                        $totalIntervals = ceil($totalDays / $intervalDays);

                        $totalProfit = ($fdr->amount * $fdr->interest_rate) / 100;
                        $profitPerInterval = $totalProfit / $totalIntervals;

                        $transaction = new Transaction();
                        $transaction->email = $user->email;
                        $transaction->amount = $profitPerInterval;
                        $transaction->type = "FDR Profit";
                        $transaction->profit = "plus";
                        $transaction->txnid = $fdr->transaction_no;
                        $transaction->user_id = $user->id;
                        $transaction->user_available_balance = $user->balance + $profitPerInterval;
                        $transaction->save();

                        $user->balance += $profitPerInterval;
                        $user->save();

                        // Move to next profit date
                        $fdr->next_profit_time = Carbon::parse($fdr->next_profit_time)->addDays($intervalDays);
                    }

                    // If matured, return principal + any remaining profit (due to rounding)
                    if ($fdr->matured_time <= $now) {
                        $principal = $fdr->amount;

                        $givenProfit = Transaction::where('txnid', $fdr->transaction_no)
                            ->where('type', 'FDR Profit')
                            ->sum('amount');

                        $totalProfit = ($fdr->amount * $fdr->interest_rate) / 100;
                        $remainingProfit = $totalProfit - $givenProfit;

                        if ($remainingProfit > 0) {
                            $profitTrans = new Transaction();
                            $profitTrans->email = $user->email;
                            $profitTrans->amount = $remainingProfit;
                            $profitTrans->type = "FDR Profit (Final)";
                            $profitTrans->profit = "plus";
                            $profitTrans->txnid = $fdr->transaction_no;
                            $profitTrans->user_id = $user->id;
                            $profitTrans->user_available_balance = $user->balance + $remainingProfit;
                            $profitTrans->save();

                            $user->balance += $remainingProfit;
                        }

                        // Return principal
                        $principalTrans = new Transaction();
                        $principalTrans->email = $user->email;
                        $principalTrans->amount = $principal;
                        $principalTrans->type = "FDR Principal Return";
                        $principalTrans->profit = "plus";
                        $principalTrans->txnid = $fdr->transaction_no;
                        $principalTrans->user_id = $user->id;
                        $principalTrans->user_available_balance = $user->balance + $principal;
                        $principalTrans->save();

                        $user->balance += $principal;
                        $user->save();

                        $fdr->status = 2;
                    }

                    $fdr->save();
                }

                elseif ($planType === 'fixed' && $fdr->matured_time <= $now) {
                    $principal = $fdr->amount;
                    $profitAmount = $fdr->profit_amount;

                    // Return profit
                    $profitTrans = new Transaction();
                    $profitTrans->email = $user->email;
                    $profitTrans->amount = $profitAmount;
                    $profitTrans->type = "FDR Profit";
                    $profitTrans->profit = "plus";
                    $profitTrans->txnid = $fdr->transaction_no;
                    $profitTrans->user_id = $user->id;
                    $profitTrans->user_available_balance = $user->balance + $profitAmount;
                    $profitTrans->save();

                    $user->balance += $profitAmount;

                    // Return principal
                    $principalTrans = new Transaction();
                    $principalTrans->email = $user->email;
                    $principalTrans->amount = $principal;
                    $principalTrans->type = "FDR Principal Return";
                    $principalTrans->profit = "plus";
                    $principalTrans->txnid = $fdr->transaction_no;
                    $principalTrans->user_id = $user->id;
                    $principalTrans->user_available_balance = $user->balance + $principal;
                    $principalTrans->save();

                    $user->balance += $principal;
                    $user->save();

                    $fdr->status = 2;
                    $fdr->save();
                }
            });
        }
    }
}