<?php

namespace App\Jobs;

use Carbon\Carbon;
use App\Models\User;
use App\Models\UserLoan;
use App\Models\Transaction;
use App\Models\InstallmentLog;
use App\Models\LoanMissedInstallment;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;

class ProcessLoanInstallments implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;


    public function handle(): void
    {
        logger('loan INstallment Working');
        $now = Carbon::now();

        $loans = UserLoan::with('plan', 'user')
            ->where('status', 1)
            ->get();

        foreach ($loans as $loan) {
            if ($loan->given_installment >= $loan->total_installment) {
                continue;
            }

            // Due for installment
            if ($now->gte($loan->next_installment)) {
                $this->processInstallment($loan);
            }
        }
    }

    private function processInstallment(UserLoan $loan)
    {
        $user = $loan->user;
        $installment = $loan->per_installment_amount;

        if ($user->balance < $installment) {
            // Record missed installment
            $this->recordMissedInstallment($loan, $installment);
            
            $userData = User::where('id', $loan->user->id)->first();
            if($userData){
                $userData->is_suspended = 1;
                $userData->suspend_reason = 'Insufficient funds for Loan Installment';
                $userData->update();
            }
            
            // Still move to next installment date but don't increment given_installment
            $loan->next_installment = Carbon::now()->addDays((float)$loan->plan->installment_interval);
            $loan->missed_installment_count += 1;
            $loan->save();
            
            return true;
        }

        $user->balance -= $installment;
        $user->save();

        $loan->given_installment += 1;
        $loan->paid_amount += $installment;

        // If not finished yet → set next installment
        if ($loan->given_installment < $loan->total_installment) {
            $loan->next_installment = Carbon::now()->addDays((float)$loan->plan->installment_interval);
        } else {
            // Check if all installments including missed ones are paid
            $unpaidMissed = $loan->unpaidMissedInstallments()->count();
            if ($unpaidMissed > 0) {
                // Don't mark as completed if there are unpaid missed installments
                $loan->next_installment = null;
            } else {
                $loan->status = 3;
            }
        }
        $loan->save();

        $instDataLog = new InstallmentLog();
        $instDataLog->user_id = $user->id;
        $instDataLog->transaction_no = $loan->transaction_no;
        $instDataLog->type = 'loan';
        $instDataLog->amount = $installment;
        $instDataLog->save();

        // Create transaction entry
        Transaction::create([
            'email'                   => $user->email,
            'amount'                  => $installment,
            'type'                    => 'Loan Installment',
            'profit'                  => 'minus',
            'txnid'                   => $loan->transaction_no,
            'user_id'                 => $user->id,
            'user_available_balance'  => $user->balance,
        ]);

        return true;
    }

    private function recordMissedInstallment(UserLoan $loan, $amount)
    {
        // Calculate which installment number this is
        $installmentNumber = $loan->given_installment + 1;
        
        LoanMissedInstallment::create([
            'user_loan_id' => $loan->id,
            'user_id' => $loan->user_id,
            'transaction_no' => $loan->transaction_no,
            'amount' => $amount,
            'installment_number' => $installmentNumber,
            'due_date' => $loan->next_installment,
            'missed_date' => Carbon::now(),
            'is_paid' => false,
        ]);
    }
}