<?php

namespace App\Http\Controllers\Admin;

use Datatables;
use App\Models\User;
use App\Models\Currency;
use App\Models\UserLoan;
use App\Models\Transaction;
use Illuminate\Http\Request;
use App\Models\InstallmentLog;
use App\Models\LoanMissedInstallment;
use Illuminate\Support\Carbon;
use App\Http\Controllers\Controller;

class LoanController extends Controller
{
    public function __construct()
    {

    }

    public function datatables(Request $request)
    {
        if ($request->status == 'all') {
            $datas = UserLoan::with('lastHistory')->orderBy('id', 'desc')->get();
        } else {
            $datas = UserLoan::with('lastHistory')->where('status', $request->status)->orderBy('id', 'desc')->get();
        }

        return Datatables::of($datas)
            ->editColumn('status_last_history', function ($data) {
                if ($data->lastHistory) {
                    $name = e($data->lastHistory->admin->name);
                    $time = $data->lastHistory->created_at;
                    return "{$name}<br><small>{$time}</small>";
                }
            })
            ->editColumn('transaction_no', function (UserLoan $data) {
                return '<div>
                                      '.$data->transaction_no.'
                                      <br>
                                      <span class="text-info">'.$data->plan->title.'</span>
                              </div>';
            })
            ->editColumn('user_id', function (UserLoan $data) {
                return '<div>
                                          <span>'.$data->user->name.'</span>
                                          <p>'.$data->user->account_number.'</p>
                                      </div>';
            })
            ->editColumn('loan_amount', function (UserLoan $data) {
                $curr = Currency::where('is_default', '=', 1)->first();
                return '<div>
                                          '.$curr->sign.$data->loan_amount.'
                                          <br>
                                          <span class="text-info">Per Installment '.$curr->sign.$data->per_installment_amount.'</span>
                                      </div>';
            })
            ->editColumn('total_installment', function (UserLoan $data) {
                $missedInfo = '';
                if ($data->missed_installment_count > 0) {
                    $missedInfo = '<br><span class="text-danger">'.$data->missed_installment_count.' Missed</span>';
                }
                return '<div>
                                      '.$data->total_installment.'
                                      <br>
                                      <span class="text-info">'.$data->given_installment.' Given</span>
                                      '.$missedInfo.'
                              </div>';
            })
            ->editColumn('total_amount', function (UserLoan $data) {
                $curr = Currency::where('is_default', '=', 1)->first();
                return '<div>
                                          '.$curr->sign.round($data->total_installment * $data->per_installment_amount, 2).'
                                          <br>
                                          <span class="text-info">Paid Amount '.$curr->sign.$data->paid_amount.'</span>
                                      </div>';
            })
            ->editColumn('next_installment', function (UserLoan $data) {
                return $data->next_installment ? Carbon::parse($data->next_installment)->toDateString() : '--';
            })
            ->addColumn('status', function (UserLoan $data) {
                if ($data->status == 0) {
                    $status = __('Pending');
                    $status_sign = 'warning';
                } elseif ($data->status == 1) {
                    $status = __('Running');
                    $status_sign = 'success';
                } elseif ($data->status == 3) {
                    $status = __('Completed');
                    $status_sign = 'info';
                } else {
                    $status = __('Rejected');
                    $status_sign = 'danger';
                }

                if ($data->status == 3) {
                    return '<div class="btn-group mb-1">
                                <span class="badge bg-'.$status_sign.'">'.$status.'</span>
                            </div>';
                }

                $actions = '';

                if ($data->status != 1 && $data->status != 2 && $data->status != 3) {
                    $actions .= '<a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item"
                                    data-href="'.route('admin.loan.status', ['id1' => $data->id, 'id2' => 0]).'">'.__("Pending").'</a>';
                }

                if ($data->status != 1) {
                    $actions .= '<a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item"
                                    data-href="'.route('admin.loan.status', ['id1' => $data->id, 'id2' => 1]).'">'.__("Approved").'</a>';
                }
                if ($data->status != 2) {
                    $actions .= '<a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item"
                                    data-href="'.route('admin.loan.status', ['id1' => $data->id, 'id2' => 2]).'">'.__("Rejected").'</a>';
                }

                return '<div class="btn-group mb-1">
                            <button type="button" class="btn btn-'.$status_sign.' btn-sm btn-rounded dropdown-toggle"
                                    data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                                '.$status.'
                            </button>
                            <div class="dropdown-menu" x-placement="bottom-start">
                                '.$actions.'
                            </div>
                        </div>';
            })

            ->addColumn('action', function (UserLoan $data) {
                $missedLink = '';
                if ($data->missed_installment_count > 0) {
                    $missedLink = '<a href="'.route('admin.loan.missed.installments', $data->id).'"  class="dropdown-item text-danger">'.__("Missed Installments").'</a>';
                }

                return '<div class="btn-group mb-1">
                          <button type="button" class="btn btn-primary btn-sm btn-rounded dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                            '.'Actions'.'
                          </button>
                          <div class="dropdown-menu" x-placement="bottom-start">
                            <a href="'.route('admin.loan.log.show', $data->id).'"  class="dropdown-item">'.__("Logs").'</a>
                            <a href="'.route('admin.loan.show', $data->id).'"  class="dropdown-item">'.__("Details").'</a>
                            '.$missedLink.'
                          </div>
                        </div>';

            })
            ->rawColumns(['status_last_history', 'transaction_no', 'user_id', 'loan_amount', 'total_installment', 'total_amount', 'next_installment', 'status', 'action'])
            ->toJson();
    }

    public function index()
    {
        $this->installmentCheck();
        return view('admin.loan.index');
    }

    public function running()
    {
        $this->installmentCheck();
        return view('admin.loan.running');
    }

    public function completed()
    {
        return view('admin.loan.completed');
    }

    public function pending()
    {
        return view('admin.loan.pending');
    }

    public function rejected()
    {
        return view('admin.loan.rejected');
    }

    public function status($id1, $id2)
    {
        $data = UserLoan::findOrFail($id1);

        if ($data->status == 1 && $id2 == 1) {
            return response()->json('This loan is already approved and running. Cannot approve again.');
        }

        if ($id2 == 2) {
            if ($data->given_installment > 0) {
                return response()->json('This loan already has installments paid. Cannot reject.');
            }

            if ($data->status == 1) {
                $user = User::find($data->user_id);
                if ($user) {
                    $user->balance -= $data->loan_amount;
                    $user->save();
                }

                $reversal = new Transaction();
                $reversal->email = $user->email;
                $reversal->amount = $data->loan_amount;
                $reversal->type = "Loan Reversal";
                $reversal->profit = "minus";
                $reversal->txnid = $data->transaction_no;
                $reversal->user_id = $user->id;
                $reversal->user_available_balance = $user->balance;
                $reversal->save();

                notifyUser($user->id, 'Loan Rejected',  'user.loans.index', $data->id);
            }

            statusHistory($data->id, 'loan', $data->status, 2);

            $data->status = 2;
            $data->save();

            return response()->json('Loan rejected successfully.');
        }

        // Approve logic
        if ($id2 == 1) {
            if ($user = User::find($data->user_id)) {
                $user->balance += $data->loan_amount;
                $user->save();

                $trans = new Transaction();
                $trans->email = $user->email;
                $trans->amount = $data->loan_amount;
                $trans->type = "Loan";
                $trans->profit = "plus";
                $trans->txnid = $data->transaction_no;
                $trans->user_id = $user->id;
                $trans->user_available_balance = $user->balance;
                $trans->save();
            }
            $data->next_installment = now()->addDays((float)$data->plan->installment_interval);
            statusHistory($data->id, 'loan', $data->status, 1);
            $data->status = 1;
            $data->save();

            notifyUser($user->id, 'Loan Confirmed',  'user.loans.index', $data->id);

            return response()->json('Loan approved successfully.');
        }

        $data->status = $id2;
        $data->update();

        return response()->json('Data Updated Successfully.');
    }

    public function show($id)
    {
        $data = UserLoan::with('histories')->findOrFail($id);
        $data['requiredInformations'] = json_decode($data->required_information, true);
        $data['data'] = $data;
        $data['currency'] = Currency::whereIsDefault(1)->first();

        return view('admin.loan.show', $data);
    }

    public function logShow($id)
    {
        $loan = UserLoan::findOrfail($id);
        $logs = InstallmentLog::where('transaction_no', $loan->transaction_no)->latest()->paginate(20);
        $currency = Currency::whereIsDefault(1)->first();

        return view('admin.loan.log', compact('loan', 'logs', 'currency'));
    }

    public function installmentCheck()
    {
        $loans = UserLoan::whereStatus(1)->get();
        $now = Carbon::now();

        foreach ($loans as $key => $data) {
            if ($data->given_installment == $data->total_installment) {
                return false;
            }
            if ($now->gt($data->next_installment)) {
                $this->takeLoanAmount($data->user_id, $data->per_installment_amount);
                $this->logCreate($data->transaction_no, $data->per_installment_amount, $data->user_id);

                $data->next_installment = Carbon::now()->addDays((float)$data->plan->installment_interval);
                $data->given_installment += 1;
                $data->paid_amount += $data->per_installment_amount;
                $data->update();

                if ($data->given_installment == $data->total_installment) {
                    $this->paid($data);
                }
            }
        }
    }

    public function takeLoanAmount($userId, $installment)
    {
        $user = User::whereId($userId)->first();
        if ($user && $user->balance >= $installment) {
            $user->balance -= $installment;
            $user->update();
        }
    }

    public function paid($loan)
    {
        $loan = UserLoan::whereId($loan->id)->first();
        if ($loan) {
            $loan->status = 3;
            $loan->next_installment = null;
            $loan->update();
        }
    }

    public function logCreate($transactionNo, $amount, $userId)
    {
        $data = new InstallmentLog();
        $data->user_id = $userId;
        $data->transaction_no = $transactionNo;
        $data->type = 'loan';
        $data->amount = $amount;
        $data->save();
    }

    /**
     * Show missed installments for a specific loan
     */
    public function missedInstallments($loanId)
    {
        $loan = UserLoan::with(['user', 'plan', 'missedInstallments' => function($query) {
            $query->orderBy('installment_number', 'asc');
        }])->findOrFail($loanId);
        
        $currency = Currency::whereIsDefault(1)->first();
        $unpaidMissed = $loan->unpaidMissedInstallments;
        $totalUnpaidAmount = $unpaidMissed->sum('amount');
        
        return view('admin.loan.missed_installments', compact('loan', 'currency', 'unpaidMissed', 'totalUnpaidAmount'));
    }

    /**
     * Process payment for missed installments and unsuspend user
     */
    public function payMissedInstallments(Request $request, $loanId)
    {
        $loan = UserLoan::with('user')->findOrFail($loanId);
        $user = $loan->user;
        
        // Get all unpaid missed installments for this loan
        $unpaidMissed = LoanMissedInstallment::where('user_loan_id', $loanId)
            ->where('is_paid', false)
            ->orderBy('installment_number', 'asc')
            ->get();
        
        if ($unpaidMissed->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'No unpaid missed installments found.'
            ]);
        }
        
        $totalAmount = $unpaidMissed->sum('amount');
        
        // Check if user has sufficient balance
        if ($user->balance < $totalAmount) {
            return response()->json([
                'success' => false,
                'message' => 'User does not have sufficient balance to pay missed installments. Required: ' . $totalAmount . ', Available: ' . $user->balance
            ]);
        }
        
        // Process payment
        $user->balance -= $totalAmount;
        $user->save();
        
        // Mark all missed installments as paid
        foreach ($unpaidMissed as $missed) {
            $missed->is_paid = true;
            $missed->paid_date = now();
            $missed->notes = 'Paid by admin approval on ' . now()->format('Y-m-d H:i:s');
            $missed->save();
            
            // Create installment log
            $instDataLog = new InstallmentLog();
            $instDataLog->user_id = $user->id;
            $instDataLog->transaction_no = $loan->transaction_no;
            $instDataLog->type = 'loan';
            $instDataLog->amount = $missed->amount;
            $instDataLog->save();
            
            // Create transaction
            Transaction::create([
                'email' => $user->email,
                'amount' => $missed->amount,
                'type' => 'Missed Loan Installment',
                'profit' => 'minus',
                'txnid' => $loan->transaction_no,
                'user_id' => $user->id,
                'user_available_balance' => $user->balance,
            ]);
        }
        
        // Update loan
        $loan->paid_amount += $totalAmount;
        $loan->given_installment += $unpaidMissed->count();
        $loan->missed_installment_count = 0;
        
        // Check if loan should be completed
        if ($loan->given_installment >= $loan->total_installment) {
            $loan->status = 3; // Completed
            $loan->next_installment = null;
        }
        
        $loan->save();
        
        // Unsuspend user
        if ($user->is_suspended) {
            $user->is_suspended = 0;
            $user->suspend_reason = null;
            $user->save();
        }
        
        notifyUser($user->id, 'Missed Installments Paid', 'user.loans.index', $loan->id);
        
        return response()->json([
            'success' => true,
            'message' => 'All missed installments paid successfully. User has been unsuspended.'
        ]);
    }

    /**
     * Manually unsuspend user without payment (admin decision)
     */
    public function unsuspendUser($loanId)
    {
        $loan = UserLoan::with('user')->findOrFail($loanId);
        $user = $loan->user;
        
        if (!$user->is_suspended) {
            return response()->json([
                'success' => false,
                'message' => 'User is not suspended.'
            ]);
        }
        
        $user->is_suspended = 0;
        $user->suspend_reason = null;
        $user->save();
        
        notifyUser($user->id, 'Account Unsuspended', 'user.loans.index', $loan->id);
        
        return response()->json([
            'success' => true,
            'message' => 'User unsuspended successfully.'
        ]);
    }
}
