<?php

namespace App\Http\Controllers\Admin;

use PDF;
use Datatables;
use Carbon\Carbon;
use App\Models\User;
use App\Models\Order;
use App\Models\Follow;
use App\Models\Rating;
use App\Models\UserDps;
use App\Models\UserFdr;
use App\Models\Currency;
use App\Models\UserLoan;
use App\Models\Wishlist;
use App\Models\Withdraw;
use App\Models\LoginLogs;
use App\Models\OrderedItem;
use App\Models\Transaction;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\Generalsetting;
use App\Models\PaymentGateway;
use App\Models\AdminTransaction;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Blade;
use Illuminate\Support\Facades\Validator;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:admin');
    }

    public function datatables()
    {
        $columns = ['id', 'name', 'last_name', 'email', 'phone', 'account_type', 'balance', 'bonus_point', 'status', 'is_suspended', 'suspend_reason', 'is_banned', 'email_2fa', 'twofa', 'login_attempts', 'blocked_until', 'last_session_id', 'account_number'];
        $list = User::query()->select($columns)->newQuery();
        $currency = Currency::whereIsDefault(1)->first();

        //\request()

        return Datatables::of($list)
            ->editColumn('full_name', function (User $data) {
                return $data->name . ''. $data->last_name;
            })
            ->editColumn('balance', function (User $data) use ($currency) {
                return $currency->sign . amount_format($data->balance);
            })
            ->editColumn('account_type', function (User $data) use ($currency) {
                return ucwords($data->account_type);
            })
            ->editColumn('phone', function (User $data) use ($currency) {
                return preg_replace("/[^0-9+]/", "", $data->phone);
            })

            ->editColumn('account_number', function (User $data) use ($currency) {
                return  $data->account_number;
            })


            ->addColumn('loan_amount', function (User $data) use ($currency) {
                //  return preg_replace("/[^0-9+]/", "", $data->phone);
                return $currency->sign . amount_format($data->loans->where('status', 1)->sum('loan_amount') ?? 0);
            })

            ->addColumn('dps_amount', function (User $data) use ($currency) {
                return $currency->sign . amount_format($data->dps->where('status', 1)->sum('deposit_amount') ?? 0);
            })

            ->addColumn('fdr_amount', function (User $data) use ($currency) {
                // return preg_replace("/[^0-9+]/", "", $data->phone);
                return $currency->sign . amount_format($data->fdr->where('status', 1)->sum('amount') ?? 0);
            })
            ->addColumn('action', function (User $data) {
                $btnEmail2fa = '<a href="' . route('admin-user-email2fa', [$data->id, 'enable']) . '" class="dropdown-item">' . __("Enable Email 2FA") . '</a>';
                if ($data->email_2fa == 1) {
                    $btnEmail2fa = '<a href="' . route('admin-user-email2fa', [$data->id, 'disable']) . '" class="dropdown-item">' . __("Disable Email 2FA") . '</a>';
                }
                $btnApp2fa = '<a href="' . route('admin-user-app2fa', [$data->id, 'enable']) . '" class="dropdown-item">' . __("Enable 2FA") . '</a>';
                if ($data->twofa == 1) {
                    $btnApp2fa = '<a href="' . route('admin-user-app2fa', [$data->id, 'disable']) . '" class="dropdown-item">' . __("Disable 2FA") . '</a>';
                }

                $btnUnblock = '';

                if ($data->blocked_until && now()->lessThan($data->blocked_until)) {
                    $btnUnblock = '<a href="' . route('admin.user.unblock', $data->id) . '" class="dropdown-item text-danger">' . __("Unblock User") . '</a>';
                }

                return '
                    <div class="btn-group mb-1">
                        <button type="button" class="btn btn-primary btn-sm btn-rounded dropdown-toggle"
                            data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">Actions</button>
                        <div class="dropdown-menu" x-placement="bottom-start">
                            <a href="javascript:;"  class="dropdown-item login-to-user" data-id="' . $data->id . '"
                                    data-email="' . $data->email . '" data-name="' . $data->name . '">
                                ' . __("Login to User") . '
                            </a>
                            <a href="' . route('admin-user-show', $data->id) . '" class="dropdown-item">' . __("Details") . '</a>
                            <a href="' . route('admin-user-edit', $data->id) . '" class="dropdown-item" >' . __("Edit") . '</a>
                            <a href="javascript:;" class="dropdown-item send" data-email="' . $data->email . '" data-toggle="modal" data-target="#vendorform">' . __("Send") . '</a>
                            <a href="javascript:;" data-toggle="modal" data-target="#deleteModal" class="dropdown-item" data-href="' . route('admin-user-delete', $data->id) . '">' . __("Delete") . '</a>
                            ' . $btnEmail2fa . '
                            ' . $btnApp2fa . '
                               ' . $btnUnblock . '
                        </div>
                    </div>
                ';
            })
            ->addColumn('status', function (User $data) {
                $status = $data->is_banned == 1 ? __('Block') : __('Unblock');
                $status_sign = $data->is_banned == 1 ? 'danger' : 'success';

                $button = '<a href="javascript:void(0);" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="' . route('admin-user-ban', ['id1' => $data->id, 'id2' => 1]) . '">' . __("Block") . '</a>';
                if ($data->is_banned == 1) {
                    $button = '<a href="javascript:void(0);" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="' . route('admin-user-ban', ['id1' => $data->id, 'id2' => 0]) . '">' . __("Unblock") . '</a>';
                }

                return '
                    <div class="btn-group mb-1">
                         <button type="button" class="btn btn-' . $status_sign . ' btn-sm btn-rounded dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                             ' . $status . '
                         </button>
                         <div class="dropdown-menu" x-placement="bottom-start">' . $button . '</div>
                     </div>
                ';
            })
            ->addColumn('suspend', function (User $data) {
                $status = $data->is_suspended == 1 ? __('Suspended') : __('Unsuspended');
                $status_sign = $data->is_suspended == 1 ? 'danger' : 'success';

                $tooltip = "";
                $button = '<a href="javascript:void(0);" class="dropdown-item suspend-user" data-user-id="' . $data->id . '" >' . __("Suspended") . '</a>';
                if ($data->is_suspended == 1) {
                    $button = '<a href="javascript:void(0);" class="dropdown-item unsuspend-user" data-user-id="' . $data->id . '" >' . __("Unsuspended") . '</a>';
                    $tooltip = '<i class="fa fa-info-circle" data-toggle="tooltip" data-placement="top" title="Suspend Reason: ' . addslashes($data->suspend_reason) . '" role="button"></i>';
                }

                return '
                    <div class="d-inline-flex align-items-center">
                        <div class="btn-group mb-1">
                             <button type="button" class="btn btn-' . $status_sign . ' btn-sm btn-rounded dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                                 ' . $status . '
                             </button>
                             <div class="dropdown-menu" x-placement="bottom-start">
                                 ' . $button . '
                             </div>
                         </div>
                         ' . $tooltip . '
                    </div>
                ';
            })
            ->rawColumns(['action', 'status', 'suspend'])
            ->toJson();
    }

    public function index()
    {
        return view('admin.user.index');
    }

    public function image()
    {
        return view('admin.generalsetting.user_image');
    }

    public function show($id)
    {
        $data = User::findOrFail($id);
        $data['loans'] = UserLoan::whereUserId($data->id)->get();
        $data['dps'] = UserDps::whereUserId($data->id)->get();
        $data['fdr'] = UserFdr::whereUserId($data->id)->get();
        $data['withdraws'] = Withdraw::whereUserId($data->id)->get();
        $data['data'] = $data;
        return view('admin.user.show', $data);
    }

    public function ban($id1, $id2)
    {
        $user = User::findOrFail($id1);
        $user->is_banned = $id2;
        if ($id2 == 0) $user->is_suspended = 0;
        $user->update();
        $msg = 'Data Updated Successfully.';
        if (!request()->ajax()) {
            return redirect()->back()->with('message', $msg);
        }

        return response()->json($msg);
    }


    public function edit($id)
    {
        $data = User::findOrFail($id);
        return view('admin.user.edit', compact('data'));
    }


    public function update(Request $request, $id)
    {
        $rules = [
            'photo' => 'mimes:jpeg,jpg,png,svg',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json(array('errors' => $validator->getMessageBag()->toArray()));
        }

        $user = User::findOrFail($id);
        $data = $request->all();
        if ($file = $request->file('photo')) {
            $name = Str::random(8) . time() . '.' . $file->getClientOriginalExtension();
            $file->move('assets/images', $name);
            if ($user->photo != null) {
                if (file_exists(public_path() . '/assets/images/' . $user->photo)) {
                    unlink(public_path() . '/assets/images/' . $user->photo);
                }
            }
            $data['photo'] = $name;
        }
        $user->update($data);
        //return $user;
        $msg = 'Customer Information Updated Successfully.';
        return response()->json($msg);
    }

    public function addDeduct(Request $request)
    {
        DB::beginTransaction();
        try {
            $user = User::whereId($request->user_id)->first();
            if ($request->type === 'add') {
                $user->increment('balance', $request->amount);

                AdminTransaction::create([
                    'user_id' => $user->id,
                    'add_amount' => $request->amount,
                    'sub_amount' => null,
                    'type' => 'admin',
                ]);


                $transaction = new Transaction();
                $transaction->email = $user->email;
                $transaction->amount = $request->amount;
                $transaction->type = "Admin";
                $transaction->profit = "plus";
                $transaction->txnid = NULL;
                $transaction->user_id = $user->id;
                $transaction->user_available_balance = $user->balance;
                $transaction->save();

                DB::commit();

                return redirect()->back()->with('message', 'User balance added');
            } else {
                if ($user->balance >= $request->amount) {
                    $user->decrement('balance', $request->amount);

                    AdminTransaction::create([
                        'user_id' => $user->id,
                        'add_amount' => null,
                        'sub_amount' => $request->amount,
                        'type' => 'admin',
                    ]);

                    $transaction = new Transaction();
                    $transaction->email = $user->email;
                    $transaction->amount = $request->amount;
                    $transaction->type = "Admin";
                    $transaction->profit = "minus";
                    $transaction->txnid = NULL;
                    $transaction->user_id = $user->id;
                    $transaction->user_available_balance = $user->balance;
                    $transaction->save();

                    DB::commit();

                    return redirect()->back()->with('message', 'User balance deducted!');
                } else {
                    return redirect()->back()->with('warning', 'User doesn’t have sufficient balance!');
                }
            }
        } catch (\Exception $th) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Something Went Wrong');
        }
        
    }

    public function withdraws()
    {
        $data['pending'] = Withdraw::whereStatus('pending')->count();
        $data['complete'] = Withdraw::whereStatus('completed')->count();
        $data['cancelled'] = Withdraw::whereStatus('rejected')->count();
        $data['processing'] = Withdraw::whereStatus('processing')->count();
        $data['onhold'] = Withdraw::whereStatus('hold')->count();
        return view('admin.user.withdraws', $data);
    }

    public function withdrawdatatables()
    {
        $eloquent = Withdraw::with('user', 'lastHistory')->orderBy('id', 'desc');

        return Datatables::of($eloquent)
            ->filterColumn('email', function ($query, $keyword) {
                $query->whereHas('user', function ($q) use ($keyword) {
                    $q->where('email', 'LIKE', "%$keyword%");
                });
            })
            ->filterColumn('phone', function ($query, $keyword) {
                $query->whereHas('user', function ($q) use ($keyword) {
                    $q->where('phone', 'LIKE', "%$keyword%");
                });
            })
            ->addColumn('checkbox', function (Withdraw $data) {
                return '<input type="checkbox" class="row-checkbox" value="' . $data->id . '">';
            })
            ->filterColumn('status', function ($query, $keyword) {
                $query->where('status', 'LIKE', "%$keyword%");
            })
            ->addColumn('email', function (Withdraw $data) {
                return $data->user->email;
            })
            ->addColumn('phone', function (Withdraw $data) {
                return $data->user->phone;
            })
            ->editColumn('status', function (Withdraw $row) {
                [$status_label, $status_sign] = match ($row->status) {
                    "pending" => [__("Pending"), "warning"],
                    "processing" => [__("Processing"), "warning"],
                    "completed" => [__("Completed"), "success"],
                    "hold" => [__("Hold"), "danger"],
                    "rejected" => [__("Rejected"), "danger"],
                    default => [__("Unknown"), "secondary"],
                };

                $html = <<<'HTML'
                    <div class="btn-group mb-1">
                        <button type="button" class="btn btn-{{$status_sign}} btn-sm btn-rounded dropdown-toggle"
                                data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                        {{ $status_label }}
                        </button>
                        <div class="dropdown-menu" x-placement="bottom-start">
                        <a href="#status-modal" data-toggle="modal" class="dropdown-item" data-href="{{ route('admin.withdraw.status', ['id' => $id, 'status' => 'pending']) }}">{{ __("Pending") }}</a>
                        <a href="#status-modal" data-toggle="modal" class="dropdown-item" data-href="{{ route('admin.withdraw.status', ['id' => $id, 'status' => 'processing']) }}">{{ __("Processing") }}</a>
                        <a href="#status-modal" data-toggle="modal" class="dropdown-item" data-href="{{ route('admin-withdraw-accept', $id) }}">{{ __("Completed") }}</a>
                        <a href="#status-modal" data-toggle="modal" class="dropdown-item" data-href="{{ route('admin.withdraw.status', ['id' => $id, 'status' => 'hold']) }}">{{ __("Hold") }}</a>
                        <a href="#status-modal" data-toggle="modal" class="dropdown-item" data-href="{{ route('admin-withdraw-reject', $id) }}">{{ __("Rejected") }}</a>
                        <!-- <a href="#confirm-delete" data-toggle="modal" class="dropdown-item" data-href="{{ route('admin-withdraw-reject', $id) }}">{{ __("Rejected") }}</a> -->
                        </div>
                    </div>
                HTML;

                return Blade::render($html, [
                    'id' => $row->id,
                    'status_label' => $status_label,
                    'status_sign' => $status_sign,
                ]);
            })


            ->editColumn('last_history', function ($data) {
                if ($data->lastHistory) {
                    $name = e($data->lastHistory->admin->name);
                    $time = $data->lastHistory->created_at;
                    return "{$name}<br><small>{$time}</small>";
                }
            })

            ->editColumn('amount', function (Withdraw $data) {
                $amount = $data->amount;
                return '<div class="text-right pr-4">$' . $amount . '</div>';
            })
            ->editColumn('created_at', function (Withdraw $data) {
              //  return $data->created_at->diffForHumans();
              return $data->created_at->format('d-m-Y H:i');
            })
            ->addColumn('action', function (Withdraw $data) {
                $action = "";
                return '
                    <div class="btn-group mb-1">
                        <button type="button" class="btn btn-primary btn-sm btn-rounded dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                          ' . 'Actions' . '
                        </button>
                        <div class="dropdown-menu" x-placement="bottom-start">
                            <a href="javascript:;" data-href="' . route('admin.withdraw.show', $data->id) . '"  class="dropdown-item" id="applicationDetails" data-toggle="modal" data-target="#details">' . __("Details") . '</a>
                            ' . $action . '
                        </div>
                    </div>
                ';
            })
        ->addColumn('dummy', '')
        ->rawColumns(['checkbox', 'email', 'amount', 'action', 'status', 'last_history'])
        ->toJson();
    }


    public function withdrawdetails($id)
    {
        $withdraw = Withdraw::with('histories')->findOrFail($id);
        $gateway = $withdraw->gateway;
        $currency_id = json_decode($gateway->currency_id);
        $g_currency = Currency::find($currency_id[0] ?? 0);
        $convert_rate = $withdraw->rate ?? $gateway->sending_rate;

        return view('admin.user.withdraw-details', compact('withdraw', 'gateway', 'g_currency', 'convert_rate'));
    }


    public function accept($id)
    {
        $withdraw = Withdraw::findOrFail($id);

        //removse start
        $fund = PaymentGateway::query()->whereId($withdraw->gateway_id)->first();
        $convertRate = $withdraw->rate ?? $fund->sending_rate;
        $amountToDeduct = $withdraw->amount * $convertRate;
        $fund->fund_available = $fund->fund_available - $amountToDeduct;

        //  dd($convertRate);
        //    return $fund;
        // remover latet end

        $authUser =  Auth::guard('admin')->user();; // Logged-in admin or user

        if ($withdraw->status == 'completed') {
            $msg = __('Withdraw Already Accepted.');
            return response()->json($msg);
        }
        // Check role-based permission
        if (
            !$authUser->IsSuper() &&
            ($withdraw->status == 'completed' || $withdraw->status == 'rejected')

        ) {
            // Only allow status change within 30 minutes
            // $lastUpdate = $withdraw->status_updated_at;
            // $now = Carbon::now();

             $now = Carbon::now();
              $lastUpdate = Carbon::parse($withdraw->status_updated_at);

              $diff = $lastUpdate->diffInMinutes($now, false);

            if ($diff > 30) {
                return response()->json([
                    'error' => 'You can no longer update this status. 30-minute window has passed.'
                ], 403);
            }
        }


        if ($withdraw->status == 'rejected') {
            $finalAmount = $withdraw->amount + $withdraw->fee;
            if ($withdraw->user->balance < $finalAmount) {
                $msg = __('User has insufficient balance to revert the rejected transfer.');
                return response()->json($msg);
            }

            $withdraw->user->decrement('balance', $finalAmount);

            $trans = new Transaction();
            $trans->email = $withdraw->user->email;
            $trans->amount = $withdraw->amount;
            $trans->type = "Withdraw Revert Rejected";
            $trans->profit = "minus";
            $trans->txnid = $withdraw->txnid;
            $trans->user_id = $withdraw->user_id;
            $trans->charge = $withdraw->fee;
            $trans->user_available_balance =   $withdraw->user->balance;
            $trans->receiver_available_balance = 0;
            $trans->save();

            notifyUser($withdraw->user->id, 'Withdraw Revert Rejected',  'user.withdraw.index', $withdraw->id);
        }

        $data['status'] = "completed";
        $data['status_updated_at'] =  now();
        // $data['is_fund_add'] = 1;

        statusHistory($withdraw->id, 'withdraw', $withdraw->status, 'completed');

        $withdraw->update($data);


        $fund = PaymentGateway::query()->whereId($withdraw->gateway_id)->first();
        $convertRate = $withdraw->rate ?? $fund->sending_rate;
        $amountToDeduct = $withdraw->amount * $convertRate;
        $fund->fund_available = $fund->fund_available - $amountToDeduct;
        $fund->save();
        $withdraw->is_fund_add = 1;
        $withdraw->save();

        $msg = __('Withdraw Accepted Successfully.');
        return response()->json($msg);
    }


    public function reject($id)
    {
        $withdraw = Withdraw::findOrFail($id);

        if ($withdraw->status == 'rejected') {
            $msg = __('Withdraw Already Rejected.');
            return response()->json($msg);
        }
        $authUser =  Auth::guard('admin')->user();; // Logged-in admin or user
        // Check role-based permission
        if (!$authUser->IsSuper() && ($withdraw->status == 'completed' || $withdraw->status == 'rejected')) {
            // Only allow status change within 30 minutes
            $lastUpdate = $withdraw->status_updated_at;
            $now = Carbon::now();

            if ($now->diffInMinutes($lastUpdate) > 30) {
                return response()->json(
                    'You can no longer update this status. 30-minute window has passed.'
                , 200);
            }
        }



        $account = User::findOrFail($withdraw->user->id);


        // if ($withdraw->status !== 'rejected') {
        $account->balance = $account->balance + $withdraw->amount + $withdraw->fee;
        $account->update();

        statusHistory($withdraw->id, 'withdraw', $withdraw->status, 'rejected');

        if ($withdraw->is_fund_add == 1) {
            $fund = PaymentGateway::query()->whereId($withdraw->gateway_id)->first();
            $convertRate = $withdraw->rate ?? $fund->sending_rate;
            $amountToDeduct = $withdraw->amount * $convertRate;
            $fund->fund_available = $fund->fund_available + $amountToDeduct;
            $fund->save();
            $withdraw->is_fund_add = 0;
            $withdraw->save();
        }


        $allinfo = [
            'sender_type' => "Refund",
            'reciver_type' => "Nan",
            "fixed_charge_amount" => 0,
            "percent_charge" => 0,
            "percent_charge_amount" => 0,
        ];

        $trans = new Transaction();
        $trans->email = $account->email;
        $trans->amount = $withdraw->amount;
        $trans->type = "Withdraw Refund";
        $trans->profit = "plus";
        $trans->txnid = $withdraw->txnid;
        $trans->user_id = $account->id;
        $trans->charge = $withdraw->fee;
        $trans->user_available_balance =   $account->balance;
        $trans->receiver_available_balance = 0;
        $trans->all_info = json_encode($allinfo);
        $trans->save();
        // }
        $data['status'] = "rejected";
        $data['status_updated_at'] =  now();
        $data['is_fund_add'] = 0;
        $withdraw->update($data);

        $msg = __('Withdraw Rejected Successfully.');
        return response()->json($msg);
    }

    public function statusChange($id, $status)
    {
        $withdraw = Withdraw::findOrFail($id);

        if ($withdraw->is_fund_add == 1) {
            $fund = PaymentGateway::query()->whereId($withdraw->gateway_id)->first();
            $convertRate = $withdraw->rate ?? $fund->sending_rate;
            $amountToDeduct = $withdraw->amount * $convertRate;
            $fund->fund_available = $fund->fund_available + $amountToDeduct;
            $fund->save();

            $withdraw->is_fund_add = 0;
            $withdraw->save();
        }


        if ($withdraw->status == 'rejected') {
            $finalAmount = $withdraw->amount + $withdraw->fee;
            if ($withdraw->user->balance < $finalAmount) {
                $msg = __('User has insufficient balance to revert the rejected transfer.');
                return response()->json($msg);
            }

            $withdraw->user->decrement('balance', $finalAmount);

            $trans = new Transaction();
            $trans->email = $withdraw->user->email;
            $trans->amount = $withdraw->amount;
            $trans->type = "Withdraw Revert Rejected";
            $trans->profit = "minus";
            $trans->txnid = $withdraw->txnid;
            $trans->user_id = $withdraw->user_id;
            $trans->charge = $withdraw->fee;
            $trans->user_available_balance =   $withdraw->user->balance;
            $trans->receiver_available_balance = 0;
            $trans->save();

            notifyUser($withdraw->user->id, 'Withdraw Revert Rejected',  'user.withdraw.index', $withdraw->id);
        }

        statusHistory($withdraw->id, 'withdraw', $withdraw->status, $status);

        $withdraw->status = $status;
        $withdraw->save();

        return response()->json('Status updated successfully.');
    }

    public function destroy($id)
    {
        $user = User::findOrFail($id);

        if ($user->transactions->count() > 0) {
            foreach ($user->transactions as $transaction) {
                $transaction->delete();
            }
        }

        if ($user->withdraws->count() > 0) {
            foreach ($user->withdraws as $withdraw) {
                $withdraw->delete();
            }
        }

        if ($user->deposits->count() > 0) {
            foreach ($user->deposits as $deposit) {
                $deposit->delete();
            }
        }

        if ($user->wiretransfers->count() > 0) {
            foreach ($user->wiretransfers as $transfer) {
                $transfer->delete();
            }
        }

        if ($user->loans->count() > 0) {
            foreach ($user->loans as $loan) {
                $loan->delete();
            }
        }

        if ($user->dps->count() > 0) {
            foreach ($user->dps as $dps) {
                $dps->delete();
            }
        }

        if ($user->fdr->count() > 0) {
            foreach ($user->fdr as $fdr) {
                $fdr->delete();
            }
        }

        if ($user->balanceTransfers->count() > 0) {
            foreach ($user->balanceTransfers as $balanceTransfer) {
                $balanceTransfer->delete();
            }
        }

        @unlink('/assets/images/' . $user->photo);
        $user->delete();

        $msg = 'Data Deleted Successfully.';
        return response()->json($msg);
    }

    public function adminTransaction(Request $request)
    {
        if ($request->ajax()) {
            $data = AdminTransaction::with('user')->latest('admin_transactions.id');

            return DataTables::of($data)
                ->addIndexColumn()
                ->filterColumn('name', function ($query, $keyword) {
                    $query->whereRelation('user', 'name', "%$keyword%");
                })
                ->filterColumn('phone', function ($query, $keyword) {
                    $query->whereRelation('user', 'phone', "%$keyword%");
                })
                ->editColumn('name', function ($row) {
                    return $row->user->name ?? 'N/A';
                })
                ->editColumn('phone', function ($row) {
                    return $row->user->phone ?? 'N/A';
                })
                ->addColumn('type', function ($row) {
                    return ucfirst($row->type);
                })
                ->addColumn('add_amount', function ($row) {
                    return $row->add_amount ? number_format($row->add_amount, 2) : '-';
                })
                ->addColumn('sub_amount', function ($row) {
                    return $row->sub_amount ? number_format($row->sub_amount, 2) : '-';
                })
                ->addColumn('date', function ($row) {
                    return $row->created_at->format('d M, Y h:i A');
                })
                ->rawColumns(['name', 'phone', 'type', 'add_amount', 'sub_amount', 'date'])
                ->make(true);
        }

        return view('admin.user.admin_transaction');
    }

    public function loginToUser(Request $request)
    {
        $data = $request->validate(['user_id' => 'required']);

        $currentUserId = session('admin.logged_in_user_id', Auth::guard('web')->id());

        session(['admin.impersonated_user_id' => $request->user_id]);
        session(['admin.logged_in_user_id' => $currentUserId]);

        if (Auth::guard('web')->check()) {
            Auth::guard('web')->logout();
        }
        Auth::guard('web')->loginUsingId($data['user_id']);

        return to_route('user.dashboard')
            ->with('success', 'Login to user successfully.');
    }

    public function suspend(Request $request)
    {
        $request->validate([
            "suspend_reason" => "required",
            "user_id" => "required"
        ]);

        $user = User::findOrFail($request->user_id);
        $user->is_suspended = 1;
        $user->suspend_reason = $request->suspend_reason;
        $user->update();

        return back()->with('success', 'User suspended successfully.');
    }

    public function unsuspend(Request $request)
    {
        $request->validate(["user_id" => "required"]);

        $user = User::findOrFail($request->user_id);
        $user->is_suspended = 0;
        $user->suspend_reason = null;
        $user->update();

        return back()->with('success', 'User unsuspended successfully.');
    }

    public function toggleEmail2fa($id, string $status)
    {
       // dd($id, $status);
         $user = User::find($id);
        if ($status == 'enable') {
            $user->email_2fa = 1;
        } else {
            $user->email_2fa = 0;
        }
        $user->save();

        return back()->with('success', "Email 2FA $status successfully.");
    }

    public function toggleApp2fa(User $user, string $status)
    {
        if ($status == 'enable') {
            $user->twofa = 1;
        } else {
            $user->twofa = 0;
        }
        $user->save();

        return back()->with('success', "2FA $status successfully.");
    }

    public function pdfprint(Request $request)
    {
        $gs = Generalsetting::first();
        $ids = $request->input('ids'); // m
        $data = [
            'title' => $gs->title,
            'date' => date('Y-m-d'),
        ];

        if ($ids) {
            $data['withdraws'] = Withdraw::with('gateway')
                ->whereIn('id', $ids)
                ->orderBy('id', 'desc')
                ->get();
        } else {
            $data['withdraws'] = Withdraw::with('gateway')
                ->orderBy('id', 'desc')
                ->get();
        }



        $pdf = PDF::loadView('admin.user.withdrawpdf', $data);

        return $pdf->stream(uniqid('withdraw-') . '.pdf');
    }

    public function userlogs()
    {
        $data['multipulUsersameIp'] = LoginLogs::with('user')
            ->select('ip')
            ->selectRaw('COUNT(DISTINCT user_id) as user_count')
            ->groupBy('ip')
            ->having('user_count', '>', 1)
            ->get();
        return view('admin.user.LogInfo', $data);
    }

    public function userlogssearch(Request $request)
    {
        if ($request->ajax()) {
            $data = LoginLogs::with('user')->latest('id');

            return \DataTables::of($data)
                ->addIndexColumn()

                // Filters for related user fields
                ->filterColumn('name', function ($query, $keyword) {
                    $query->whereRelation('user', 'name', 'LIKE', "%{$keyword}%");
                })
                ->filterColumn('email', function ($query, $keyword) {
                    $query->whereRelation('user', 'email', 'LIKE', "%{$keyword}%");
                })
                ->filterColumn('phone', function ($query, $keyword) {
                    $query->whereRelation('user', 'phone', 'LIKE', "%{$keyword}%");
                })

                // Direct fields filter
                ->filterColumn('ip', function ($query, $keyword) {
                    $query->where('ip', 'LIKE', "%{$keyword}%");
                })
                ->filterColumn('device', function ($query, $keyword) {
                    $query->where('device', 'LIKE', "%{$keyword}%");
                })

                ->filterColumn('browser', function ($query, $keyword) {
                    $query->where('login_logs.browser', 'LIKE', "%{$keyword}%");
                })
                ->filterColumn('location', function ($query, $keyword) {
                    $query->where(function ($q) use ($keyword) {
                        $q->where('login_logs.city', 'LIKE', "%{$keyword}%")
                            ->orWhere('login_logs.country', 'LIKE', "%{$keyword}%");
                    });
                })

                // Show values
                ->editColumn('name', fn($row) => $row->user->name ?? 'N/A')
                ->editColumn('email', fn($row) => $row->user->email ?? 'N/A')
                ->editColumn('phone', fn($row) => $row->user->phone ?? 'N/A')
                ->addColumn('ip', fn($row) => $row->ip ?? '-')
                ->addColumn('device', fn($row) => $row->device ?? '-')
                ->addColumn('browser', fn($row) => $row->browser ?? '-')
                ->addColumn('location', function ($row) {
                    return trim(($row->city ?? '') . ', ' . ($row->country ?? ''), ', ') ?: '-';
                })
                ->addColumn('coordinates', function ($row) {
                    return ($row->latitude && $row->longitude)
                        ? "{$row->latitude}, {$row->longitude}"
                        : '-';
                })
                ->addColumn('date', fn($row) => '<p>' . $row->created_at->format('d/m/Y h:i a') . '</p>')

                ->rawColumns(['date'])
                ->make(true);
        }

        return view('admin.user.LogInfo');
    }

    public function unblockUser($id)
    {
        $user = User::findOrFail($id);

        $user->update([
            'login_attempts' => 0,
            'blocked_until' => null,
        ]);

        return back()->with('success', "User unblocked updated successfully.");
        // return response()->json(['msg' => 'User unblocked updated successfully.']);
        // return back()->with('success', 'User unblocked successfully.');
    }
}
