<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\BankPlan;
use App\Models\Currency;
use App\Models\PaymentGateway;
use App\Models\Transaction;
use App\Models\Withdraw;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Str;

class WithdrawController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:web');
    }

    public function index()
    {
        $withdraws = Withdraw::whereUserId(auth()->id())->orderBy('id', 'desc')->paginate(10);
        return view('user.withdraw.index', compact('withdraws'));
    }

    public function create()
    {
        $data['sign'] = Currency::whereIsDefault(1)->first();
        $data['methods'] = PaymentGateway::whereWithdrawStatus(1)->orderBy('sorting_order')->get();
        $data['fixed']   = 0.00;
        $data['percent'] = 0.00; 

        return view('user.withdraw.create', $data);
    }

    public function getWithdrawFee($id)
    {
        $withdrawMethod = PaymentGateway::find($id);
        return response()->json(['fixed' => $withdrawMethod->withdraw_fee_fixed, 'percent' => $withdrawMethod->withdraw_fee_percentage]);
    }


    public function store(Request $request)
    {
        $request->validate(['amount' => 'required|gt:0']);

        $user = auth()->user();

        if ($user->bank_plan_id === null) {
            return redirect()->back()->with('unsuccess', 'You have to buy a plan to withdraw.');
        }

        if (now()->gt($user->plan_end_date)) {
            return redirect()->back()->with('unsuccess', 'Plan Date Expired.');
        }

        $bank_plan = BankPlan::whereId($user->bank_plan_id)->first();
        $dailyWithdraws = Withdraw::whereDate('created_at', '=', date('Y-m-d'))->whereStatus('completed')->sum('amount');
        $monthlyWithdraws = Withdraw::whereMonth('created_at', '=', date('m'))->whereStatus('completed')->sum('amount');

        if ($dailyWithdraws > $bank_plan->daily_withdraw) {
            return redirect()->back()->with('unsuccess', 'Daily withdraw limit over.');
        }

        if ($monthlyWithdraws > $bank_plan->monthly_withdraw) {
            return redirect()->back()->with('unsuccess', 'Monthly withdraw limit over.');
        }

        $withdrawMethod = PaymentGateway::whereId($request->methods)->first();
        $charge = $withdrawMethod->withdraw_fee_fixed;


        if ($request->amount < $withdrawMethod->min_limit) {
            return redirect()->back()->with('unsuccess', "Minimum withdraw amount is $withdrawMethod->min_limit");
        }

        if ($request->amount > $withdrawMethod->max_limit) {
            return redirect()->back()->with('unsuccess', "Maximum withdraw limit is $withdrawMethod->max_limit");
        }

        if ($request->amount > $user->balance) {
            return redirect()->back()->with('unsuccess', 'Insufficient Account Balance.');
        }

        $messageFee = (($withdrawMethod->withdraw_fee_percentage / 100) * $request->amount) + $charge;
        $messageFinal = $request->amount - $messageFee;

        $currency = Currency::whereId($request->currency_id)->first();
        $amountToAdd = $request->amount / max($currency->value, 1);

        $amount = $amountToAdd;
        $fee = (($withdrawMethod->withdraw_fee_percentage / 100) * $amount) + $charge;
        // $curency_fee = $fee  / max($currency->value, 1);
        $finalAmount = $amount - $fee;

        if ($finalAmount < 0) {
            return redirect()->back()->with('unsuccess', 'Request Amount should be greater than this ' . $amountToAdd . ' (USD)');
        }

        if ($finalAmount > $user->balance) {
            return redirect()->back()->with('unsuccess', 'Insufficient Balance.');
        }

        $finalAmount = number_format((float) $finalAmount, 2, '.', '');

        $user->balance = $user->balance - $finalAmount - $fee;
        $user->update();


        $txnid = Str::random(12);
        $newwithdraw = new Withdraw();
        $newwithdraw['user_id'] = auth()->id();
        $newwithdraw['method'] = $withdrawMethod->name ?? $withdrawMethod->title;
        $newwithdraw['gateway_id'] = $withdrawMethod->id;
        $newwithdraw['txnid'] = $txnid;

        $newwithdraw['amount'] = $finalAmount;
        $newwithdraw['rate'] = $withdrawMethod->sending_rate;
        $newwithdraw['fee'] = $fee;
        $newwithdraw['details'] = $request->details;
        $newwithdraw->save();

        $total_amount = $newwithdraw->amount + $newwithdraw->fee;

        $allinfo = [
            'sender_type' => $user->account_type ?? "",
            'reciver_type' => "Nan",
            "fixed_charge_amount" => $withdrawMethod->withdraw_fee_fixed,
            "percent_charge" => $withdrawMethod->withdraw_fee_percentaget ?? 0,
            "percent_charge_amount" => ($withdrawMethod->withdraw_fee_percentage / 100) * $amount,
        ];

        $trans = new Transaction();
        $trans->email = $user->email;
        $trans->amount = $finalAmount;
        $trans->type = "Withdraw";
        $trans->profit = "minus";
        $trans->txnid = $txnid;
        $trans->user_id = $user->id;
        $trans->charge = $fee;
        $trans->user_available_balance =   $user->balance;
        $trans->receiver_available_balance = 0;
        $trans->all_info = json_encode($allinfo);
        $trans->save();

        return redirect()->back()->with('success', 'Withdraw Request Amount : ' . $request->amount . ' Fee : ' . $messageFee . ' = ' . $messageFinal . ' (' . $currency->name . ') Sent Successfully.');
    }

    public function details(Request $request, $id)
    {
        $data['data'] = Withdraw::findOrFail($id);
        $data['currency'] = Currency::whereIsDefault(1)->first();

        return view('user.withdraw.details', $data);
    }
}
