<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Deposit extends Model
{
    protected $fillable = [
        'user_id',
        'deposit_number',
        'amount',
        'currency_id',
        'txnid',
        'method',
        'charge_id',
        'status',
        'additional_data',
        'gateway_id',
        'status_updated_at',
        'charge',
        'is_fund_add',
        'is_amount_add'
    ];

    protected $casts = [
        'additional_data' => 'array'
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class)->withDefault();
    }

    public function gateway(): BelongsTo
    {
        return $this->belongsTo(PaymentGateway::class, 'gateway_id');
    }

    public function fileUrl($file): string
    {
        return url("assets/images/deposits/$file");
    }

    protected static function booted(): void
    {

        static::created(function ($deposit) {
            $method = $deposit->gateway?->getTitle();
            if (empty($method)) {
                $method = $deposit->method;
            }
            $currency = Currency::whereIsDefault(true)->first();
            $amount = number_format(round($deposit->amount, 2), 2);
            notifyAdmins("{$deposit->user->name} has deposited {$amount} {$currency->name} via {$method}", 'deposit');
        });
    }

    public function lastHistory()
    {
            return $this->hasOne(StatusChangeHistory::class, 'model_id')
                ->where('type', 'deposit')
                ->with('admin:id,name')
                ->latest('created_at');
    }


    public function histories()
    {
        return $this->hasMany(StatusChangeHistory::class, 'model_id')->where('type', 'deposit')->with('admin:id,name')->orderBy('created_at', 'desc');
    }
}
